# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from marshmallow import Schema, fields, validates_schema, ValidationError

from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.avia_data.models import SimilarDirection

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import ModelField, Related, SettlementKey
from travel.avia.backend.main.api.register import get_handler
from travel.avia.backend.main.lib.geo import has_airport
from .settlement import get_settlement, fill_related_fields as fill_settlement_related_fields

DEFAULT_LIMIT = 8


class SimilarDirectionsParams(Schema):
    from_id = fields.Int()
    to_id = fields.Int()

    from_key = SettlementKey()
    to_key = SettlementKey()

    from_point = ModelField(model=Settlement)
    to_point = ModelField(model=Settlement)

    limit = fields.Int(missing=DEFAULT_LIMIT)
    only_with_airports = fields.Bool(required=False, missing=False)

    @validates_schema
    def not_empty(self, data):
        if all(not data.get(i) for i in ['from_id', 'from_key', 'from_point']):
            raise ValidationError('from_id or from_key required', 'from_id')

        if all(not data.get(i) for i in ['to_id', 'to_key', 'to_point']):
            raise ValidationError('to_id or to_key required', 'to_id')


class SimilarDirectionSchema(TypeSchema):
    from_point = Related(handler=get_handler('settlement'), params={'settlement': 'similar_from'})
    to_point = Related(handler=get_handler('settlement'), params={'settlement': 'similar_to'})


class SimilarDirectionsHandler(ApiHandler):
    PARAMS_SCHEMA = SimilarDirectionsParams
    TYPE_SCHEMA = SimilarDirectionSchema
    MULTI = True

    def preprocess_fields(self, fields):
        if not fields:
            return ['from_point', 'to_point']

        return fields

    def process(self, params, fields):
        from_point = params.get('from_point')
        to_point = params.get('to_point')

        if not from_point:
            from_id = params.get('from_id')
            if not from_id and params.get('from_key'):
                from_id = params.get('from_key')[1:]
            from_point = get_settlement(from_id)

        if not to_point:
            to_id = params.get('to_id')
            if not to_id and params.get('to_key'):
                to_id = params.get('to_key')[1:]
            to_point = get_settlement(to_id)

        directions = SimilarDirection.objects.filter(
            direction="%s_%s" % (from_point.point_key, to_point.point_key)
        )[:(params.get('limit') or DEFAULT_LIMIT)]

        settlement_ids = set([d.similar_to_id for d in directions])
        settlements = {
            s.id: s
            for s in list(Settlement.objects.filter(id__in=settlement_ids))
            if not params['only_with_airports'] or has_airport(s)
        }

        directions = [d for d in directions if d.similar_to_id in settlements]

        for direction in directions:
            direction.similar_to = settlements.get(direction.similar_to_id)

        fill_settlement_related_fields([d.similar_to for d in directions])

        return directions
