# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import flask
from cachetools.func import lru_cache
from marshmallow import Schema, fields, validates_schema, ValidationError, validate
from django.conf import settings
from django.utils.translation import get_language

from travel.avia.library.python.common.models.geo import Settlement, Station

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import Related, ModelField, SettlementKey, StationKey
from travel.avia.backend.main.api.register import get_handler
from .settlement import get_settlement

from unidecode import unidecode


@lru_cache(maxsize=None)
def _get_settlement_stations(city):
    return city.get_stations_by_type(use_water=True)


def get_country(station):
    return station.translocal_country(flask.g.national_version)


def get_airports(city):
    stations = _get_settlement_stations(city)
    airports = stations['plane']['stations'] + stations['plane']['related']

    return [s for s, terminals in airports]


def get_aeroexpress(city):
    stations = _get_settlement_stations(city)
    airports = stations['plane']['stations'] + stations['plane']['related']
    aeroexpresses = [s for s, _ in airports if 'aeroex' in s.type_choices_set]

    return aeroexpresses


def get_trains(city):
    stations = _get_settlement_stations(city)
    trains = stations['train']['stations'] + stations['train']['related']

    return [train for train, _ in trains]


def get_station(code):
    if unicode(code).isdigit():
        station = Station.hidden_manager.get(pk=code)
    else:
        raise NotImplementedError

    return station


def get_station_country(station, context=None):
    if station.country:
        return station.country

    if station.settlement and station.settlement.country:
        return station.settlement.country

    return None


def get_url_title(s):
    lang, _ = settings.DOMAIN_LANGUAGE_MAP.get(flask.g.national_version, 'en')
    return unidecode(s.L_title(lang=lang)).replace("'", '')


class StationParams(Schema):
    id = fields.Integer()
    key = StationKey()
    station = ModelField(model=Station)

    @validates_schema
    def not_empty(self, data):
        if all(not data.get(i) for i in ['id', 'key', 'station']):
            raise ValidationError('id or key required', 'id')


class StationTypeSchema(TypeSchema):
    prefix = fields.Str(attribute='L_prefix')
    title = fields.Str(attribute='L_name')


class StationSchema(TypeSchema):
    id = fields.Int()
    title = fields.Str(attribute='L_title')
    url_title = fields.Function(get_url_title)
    key = fields.Str(attribute='point_key')
    code = fields.Function(lambda s: s.iata or s.sirena_id)
    geo_id = fields.Int(attribute='_geo_id')
    iata_code = fields.Str(attribute='iata')
    popular_title = fields.Str(attribute='L_popular_title')
    phrase_to = fields.Method('get_phrase_to')
    phrase_from = fields.Method('get_phrase_from')
    phrase_in = fields.Function(lambda s: s.L_title(case='phrase_in'))
    t_type = fields.Method('get_transport_type_code')
    station_type = fields.Nested(StationTypeSchema)
    longitude = fields.Str(attribute='longitude')
    latitude = fields.Str(attribute='latitude')
    time_zone = fields.Str(attribute='time_zone')
    time_zone_utc_offset = fields.Method('get_time_zone_utc_offset')

    settlement = Related(handler=get_handler('settlement'), params={'settlement': 'settlement'})
    country = Related(handler=get_handler('country'), params={'country': get_station_country})

    def get_time_zone_utc_offset(self, s):
        # type: (Station) -> int
        return s.utcoffset() / 60

    def get_phrase_to(self, s):
        if get_language() == 'ru':
            phrase_to = s.L_title(case='accusative')
        else:
            phrase_to = s.L_title(case='phrase_to')

        return phrase_to

    def get_phrase_from(self, s):
        if get_language() == 'ru':
            phrase_from = s.L_title(case='genitive')
        else:
            phrase_from = s.L_title(case='phrase_from')

        return phrase_from

    def get_transport_type_code(self, s):
        return s.t_type.code


class StationHandler(ApiHandler):
    PARAMS_SCHEMA = StationParams
    TYPE_SCHEMA = StationSchema

    def preprocess_fields(self, fields):
        if not fields:
            return ['id', 'key', 'title']

        return fields

    def process(self, params, fields):
        if params.get('station'):
            return params.get('station')

        station_id = params.get('id')

        if not station_id:
            key = params.get('key')
            station_id = key[1]

        return get_station(station_id)


class StationsParams(Schema):
    settlement_id = fields.Int()
    settlement_key = SettlementKey()
    ttype = fields.Str(required=True, validate=validate.OneOf(
        ['airport', 'train', 'aeroexpress']
    ))

    settlement = ModelField(model=Settlement)

    @validates_schema
    def not_empty(self, data):
        if all(not data.get(i) for i in ['settlement_id', 'settlement_key', 'settlement']):
            raise ValidationError('id or key required', 'settlement_id')


class StationsHandler(ApiHandler):
    PARAMS_SCHEMA = StationsParams
    TYPE_SCHEMA = StationSchema
    MULTI = True

    def preprocess_fields(self, fields):
        if not fields:
            return ['id', 'key', 'title']

        return fields

    def process(self, params, fields):
        if params.get('settlement'):
            settlement = params.get('settlement')
        else:
            settlement_id = params.get('settlement_id')

            if not settlement_id:
                key = params.get('settlement_key')
                settlement_id = key[1]

            settlement = get_settlement(settlement_id)

        ttype = params.get('ttype')

        if ttype == 'airport':
            return get_airports(settlement)

        if ttype == 'aeroexpress':
            return get_aeroexpress(settlement)

        if ttype == 'train':
            return get_trains(settlement)
