# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import flask
from marshmallow import Schema, fields

from travel.avia.library.python.avia_data.models import TopFlight

from travel.avia.backend.main.api.api_schema import TypeSchema
from travel.avia.backend.main.api.api_handler import ApiHandler
from travel.avia.backend.main.api.fields import PointKey
from .settlement import get_settlement


def get_top_flights(from_key, to_key, date_time, limit=100, national_version=None):
    national_version = national_version or flask.g.get('national_version')

    queryset = TopFlight.objects.filter(
        national_version=national_version,
        from_point_key=from_key,
        to_point_key=to_key,
        day_of_week=date_time.weekday(),
    )

    return queryset.order_by('-redirects')[:limit]


class TopFlightsParams(Schema):
    from_key = PointKey(required=True)
    to_key = PointKey(required=True)
    date = fields.Date(required=True)
    limit = fields.Int(missing=10)


class TopFlightSchema(TypeSchema):
    numbers = fields.Function(lambda c: c.flights.replace(';', '-'))
    redirects = fields.Int()


class TopFlightsHandler(ApiHandler):
    PARAMS_SCHEMA = TopFlightsParams
    TYPE_SCHEMA = TopFlightSchema
    MULTI = True

    def process(self, params, fields):
        from_settlement = get_settlement(params['from_key'][1])
        to_settlement = get_settlement(params['to_key'][1])

        if not from_settlement or not to_settlement:
            return []

        return get_top_flights(
            from_settlement.point_key, to_settlement.point_key,
            params['date'], params['limit']
        )
