# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import time
import ujson as json
from functools import wraps
from logging import getLogger

import flask
import django
from django import db as django_db
from django.conf import settings
from flask import Flask, Response, request

from travel.avia.backend.main.api_types.air_traffic_recovery import AirTrafficRecoveryStatHandler
from travel.avia.backend.main.api_types.client_settlement import ClientSettlementHandler
from travel.avia.backend.main.api_types.dohopPartners import DohopVendorsHandler
from travel.avia.backend.main.api_types.partnerka import (
    partner_statuses, enable_partner, disable_partner,
    get_partner_users, get_partner_user_by_login,
    update_unavailability_rule, create_partner, create_helper, delete_helper,
    get_detailed_statistics, get_detailed_statistics_new_scheme,
    get_partner_login, create_partner_user_handler,
)
from travel.avia.backend.main.api.register import get_handler, register_types
from travel.avia.backend.main.health import available_ping_checks, HealthError
from travel.avia.backend.main.lib.file_flag import is_temporary_flag_up
from travel.avia.backend.main.middleware import HTTPSchemeProxy, Now
from travel.avia.backend.main.lib.basic_auth import BasicAuth
from travel.avia.backend.main.lib.geo import get_settlement_by_geoid
from travel.avia.backend.main.localize import localize
from travel.avia.backend.main.rest.routes import add_rest_routes
from travel.avia.backend.main.utils import get_billing_platform_by_platform
from travel.avia.backend.main.lib.access import access_jsonrpc_log, error_jsonrpc_log, timeline_jsonrpc_log
from travel.avia.backend.main.lib.exceptions import ApiException

# Импортируем типы явно, чтобы было понятно где искать
from travel.avia.backend.main.api_types.balance import BalanceHandler
from travel.avia.backend.main.api_types.best_offers import BestOffersHandler
from travel.avia.backend.main.api_types.bug_mail import BugMailHandler
from travel.avia.backend.main.api_types.company import AirlineHandler
from travel.avia.backend.main.api_types.country import CountryHandler
from travel.avia.backend.main.api_types.country_directions import CountryDirectionsHandler
from travel.avia.backend.main.api_types.currency import CurrencyHandler
from travel.avia.backend.main.api_types.direction import DirectionHandler
from travel.avia.backend.main.api_types.geo_cities import GeoCitiesHandler
from travel.avia.backend.main.api_types.geo_countries import GeoCountriesHandler
from travel.avia.backend.main.api_types.geo_lookup import GeoLookupHandler
from travel.avia.backend.main.api_types.good_price import GoodPricesHandler
from travel.avia.backend.main.api_types.info import InfoHandler
from travel.avia.backend.main.api_types.near_directions import (
    NearDirectionsHandler, NearDistancesHandler, NearestSettlementsHandler,
)
from travel.avia.backend.main.api_types.neighbour_country_directions import NeighbourCountryDirectionsHandler
from travel.avia.backend.main.api_types.partners import PartnerHandler
from travel.avia.backend.main.api_types.partnersData import PartnersDataHandler
from travel.avia.backend.main.api_types.wizardPartners import WizardPartnersHandler
from travel.avia.backend.main.api_types.partnersLogins import PartnersLoginsHandler
from travel.avia.backend.main.api_types.price import MinPricesHandler
from travel.avia.backend.main.api_types.recipe import RecipeHandler, RecipesHandler
from travel.avia.backend.main.api_types.recipe_offers import RecipeOffersHandler, RecipeOfferHandler
from travel.avia.backend.main.api_types.reviews import ReviewListHandler, AirlineReviewsHandler
from travel.avia.backend.main.api_types.schedule import ScheduleHandler
from travel.avia.backend.main.api_types.settlement import SettlementHandler
from travel.avia.backend.main.api_types.settlement_image import SettlementImageHandler
from travel.avia.backend.main.api_types.similar_directions import SimilarDirectionsHandler
from travel.avia.backend.main.api_types.station import StationHandler, StationsHandler
from travel.avia.backend.main.api_types.top_flight import TopFlightsHandler
from travel.avia.backend.main.api_types.flight_numbers_to_company_ids import FlightNumbersToCompanyIdsHandler

log = getLogger(__name__)

django.setup()

register_types((
    ('airline', AirlineHandler),
    ('airlineReviews', AirlineReviewsHandler),
    ('balance', BalanceHandler),
    ('bestOffers', BestOffersHandler),
    ('bugMail', BugMailHandler),
    ('clientSettlement', ClientSettlementHandler),
    ('country', CountryHandler),
    ('countryDirections', CountryDirectionsHandler),
    ('currency', CurrencyHandler),
    ('direction', DirectionHandler),
    ('dohopVendors', DohopVendorsHandler),
    ('geoCities', GeoCitiesHandler),
    ('geoCountries', GeoCountriesHandler),
    ('geoLookup', GeoLookupHandler),
    ('goodPrice', GoodPricesHandler),
    ('info', InfoHandler),
    ('minPrices', MinPricesHandler),
    ('nearDirections', NearDirectionsHandler),
    ('nearDistances', NearDistancesHandler),
    ('nearestSettlements', NearestSettlementsHandler),
    ('neighbourCountryDirections', NeighbourCountryDirectionsHandler),
    ('partner', PartnerHandler),
    ('partnersData', PartnersDataHandler),
    ('wizardPartners', WizardPartnersHandler),
    ('partnersLogins', PartnersLoginsHandler),
    ('recipe', RecipeHandler),
    ('recipeOffer', RecipeOfferHandler),
    ('recipeOffers', RecipeOffersHandler),
    ('recipes', RecipesHandler),
    ('reviewList', ReviewListHandler),
    ('schedule', ScheduleHandler),
    ('settlement', SettlementHandler),
    ('settlementImage', SettlementImageHandler),
    ('similarDirections', SimilarDirectionsHandler),
    ('station', StationHandler),
    ('stations', StationsHandler),
    ('topFlights', TopFlightsHandler),
    ('flightNumbersToCompanyIds', FlightNumbersToCompanyIdsHandler),
    ('airTrafficRecoveryStat', AirTrafficRecoveryStatHandler),
))


def return_json(obj, status=200):
    return Response(
        response=json.dumps(obj, ensure_ascii=False),
        status=status,
        mimetype="application/json"
    )


def save_host():
    flask.g.front_host = request.args.get('front_host', '')


def save_user_info():
    flask.g.yandexuid = request.args.get('yandexuid', None)
    flask.g.passportuid = request.args.get('passportuid', None)
    flask.g.userip = request.args.get('userip', None)
    flask.g.user_geoid = request.args.get('user_geoid', None)
    flask.g.clid = request.args.get('clid', '')
    flask.g.stid = request.args.get('stid', '')
    user_city = None
    if flask.g.user_geoid:
        user_city = get_settlement_by_geoid(flask.g.user_geoid)

    flask.g.user_city = user_city
    platform = request.args.get('platform', None)
    flask.g.billing_platform = get_billing_platform_by_platform(platform)


def add_partnerka_routes(app):
    partnerka_auth = BasicAuth(
        settings.PARTNERKA_USERNAME, settings.PARTNERKA_PASSWORD
    )

    def wrap_view(partnerka_view, rule, methods):
        @app.route(rule, methods=methods)
        @partnerka_auth.requires_auth
        @wraps(partnerka_view)
        def wrapped_view():
            try:
                return return_json({
                    'success': True,
                    'data': partnerka_view(
                        params=request.values or request.json or {}
                    )
                })
            except Exception as exc:
                log.exception('Partnerka error: %r', exc)
                return return_json({
                    'success': False,
                    'error': repr(exc),
                })

    wrap_view(partner_statuses, '/partners/statuses', methods=['GET'])
    wrap_view(enable_partner, '/partners/enable', methods=['POST'])
    wrap_view(disable_partner, '/partners/disable', methods=['POST'])
    wrap_view(get_partner_users, '/partners/users', methods=['GET'])
    wrap_view(get_partner_user_by_login, '/partners/userByLogin', methods=['GET'])
    wrap_view(update_unavailability_rule, '/partner/updateUnavailabilityRule', methods=['POST'])
    wrap_view(create_partner, '/partners/create', methods=['POST'])
    wrap_view(create_helper, '/partners/createHelper', methods=['POST'])
    wrap_view(delete_helper, '/partners/deleteHelper', methods=['POST'])
    wrap_view(get_partner_login, '/partners/getLogin', methods=['GET'])
    wrap_view(get_detailed_statistics, '/detailedStatistics', methods=['GET'])
    wrap_view(get_detailed_statistics_new_scheme, '/detailedStatisticsNewScheme', methods=['GET'])
    wrap_view(create_partner_user_handler, '/partners/createUser', methods=['POST'])


class App(Flask):
    def __init__(self, *args, **kwargs):
        super(App, self).__init__(*args, **kwargs)
        self.shutdown_flag = None
        self.cache_readiness_flag = None
        self.shared_cache_readiness_flag = None


def create_app(config=None):
    app = App(__name__)

    if config:
        app.config.update(config)

    app.wsgi_app = HTTPSchemeProxy(app.wsgi_app)
    app.wsgi_app = Now(app.wsgi_app)

    app.secret_key = 'WnDhIuzsJEBYLWLUojvQrfSnDMwYOXWK'
    app.debug = settings.DEBUG

    @app.route('/ping', methods=['GET'])
    def ping():
        if app.shutdown_flag.is_set():
            return 'Stopped', 503
        if is_temporary_flag_up(settings.PAUSE_PING_RESPONSES_FLAGFILE):
            return 'Restarting', 503
        try:
            if not app.cache_readiness_flag.is_set():
                return 'readiness_flag is not set', 503
            if not app.shared_cache_readiness_flag.is_set():
                return 'shared_cache_readiness_flag is not set', 503
            for check in available_ping_checks:
                if request.args.get(check.disable_flag) is None:
                    check.checker.check()
        except HealthError as e:
            log.warn(e, exc_info=True)
            return 'error', 503
        except Exception:
            log.exception('ping backend exception')
            return 'error', 500

        return 'ok', 200

    @app.route('/shutdown', methods=['POST'])
    def shutdown():
        try:
            app.shutdown_flag.set()
        except Exception:
            log.exception('Can not stop ping responses')
            return 'error', 500
        else:
            return 'ok', 200

    @app.route('/version', methods=['GET'])
    def version():
        return Response(settings.PKG_VERSION, mimetype='text/plain')

    @app.route('/', methods=['POST'])
    def endpoint():
        start = time.time()

        response = None
        response_data = None
        try:
            response_data, status_code = _endpoint()
            response = return_json(response_data, status_code)
        except Exception as e:
            log.exception('Caught unknown error while processing request')
            response_data = {
                'status': 'error',
                'reason': 'Unknown error: %s' % e.message
            }

            response = return_json(response_data, status=500)

        error_jsonrpc_log(request, response_data)
        timeline_jsonrpc_log(request, time.time() - start)

        return response

    def _endpoint():
        access_jsonrpc_log(request)
        localize()  # устанавливаем язык и нац версию глобально
        save_host()
        save_user_info()  # куки, ip и тп пользователя

        result = []

        if not isinstance(request.json, list):
            return {
                'status': 'error',
                'reason': 'Body must be array',
            }, 400

        for item in request.json:
            name = item['name']
            params = item.get('params', {})
            fields = item.get('fields', [])

            handler_class = get_handler(name)
            try:
                # В RASPTICKETS-17122 было решено, что будем останавливаться
                # в случае первой ошибки и выкидывать ее
                handler_result = handler_class()(params, fields)
                result.append(handler_result)
            except ApiException as e:
                return e.dump(), 400

        return {
            'status': 'success',
            'data': result
        }, 200

    add_partnerka_routes(app)
    add_rest_routes(app)

    app.before_request(lambda *_args, **_kwargs: django_db.reset_queries())

    if config and not config.get('not_close_connection'):
        app.teardown_request(lambda *_args, **_kwargs: django_db.close_connection())

    return app


if __name__ == '__main__':
    app = create_app()
    app.run(debug=True, port=8003)
