from __future__ import absolute_import

import struct
from logging import getLogger

from travel.avia.library.python.common.models.schedule import Company

from django.core.management.base import BaseCommand

HTML_PATH = '/tmp/company_colors.html'
DARK_COLOR = 'rgba(0, 0, 0, .6)'
LIGHT_COLOR = 'rgba(255, 255, 255, .8)'

log = getLogger(__name__)


class Command(BaseCommand):
    def handle(self, *args, **options):
        render_colors_html(Company.objects.exclude(logo_bgcolor=''))


def render_colors_html(company_list):
    with open(HTML_PATH, 'w') as f:
        f.write("""
            <!DOCTYPE html>
            <html>
                <head>
                    <meta charset="utf-8">
                    <title>Logo BG Colors</title>
                    <style>
                        body {
                            display: flex;
                            flex-wrap: wrap;
                            background-color: #ece8dd;
                        }
                        div {
                            padding: 10px;
                            margin: 2px;
                            font: 12px Arial;
                            flex-grow: 1;
                        }
                    </style>
                </head>
                <body>
        """)

        for company in company_list:
            render_company(company, f)

        f.write("""
                </body>\
            </html>\
        """)

        log.info('HTML file with colors: {html_path} created'.format(html_path=HTML_PATH))


def render_company(company, f):
    lightness = get_color_lightness(company.logo_bgcolor)

    color = DARK_COLOR
    if lightness < 128:
        color = LIGHT_COLOR
        log.info(str(company.id) + ', bg_color: ' + company.logo_bgcolor + ' [dark]')
    else:
        log.info(str(company.id) + ', bg_color: ' + company.logo_bgcolor + ' [light]')

    f.write("""
        <div style="background-color: {bg_color}; color: {color};">
            {title}
        </div>
    """.format(bg_color=company.logo_bgcolor, color=color, title=company.title.encode('utf-8')))


def get_color_lightness(color):
    h = color.lstrip('#')
    rgb = struct.unpack('BBB', h.decode('hex'))
    return sum(rgb) / 3
