from __future__ import absolute_import

import struct
import tempfile
import urllib2
from collections import Counter
from logging import getLogger
from optparse import make_option

import cairo
import rsvg
from PIL import Image

from django.core.management.base import BaseCommand

from travel.avia.library.python.common.models.schedule import Company

LOGO_SIZE = 30
TRANSPARENT_COLOR = (255, 255, 255)
HTML_PATH = '/tmp/company_logos.html'

log = getLogger(__name__)


class Command(BaseCommand):
    help = 'updates the colors of the companies on the basis of the color of their logo'

    option_list = BaseCommand.option_list + (
        make_option('--render-html', action='store_true', default=False,
                    help='Render html with companies\' logos'),
        make_option('--force', action='store_true', default=False,
                    help='Force update all colors'),
    )

    def handle(self, *args, **options):
        company_model_list = Company.objects.exclude(sirena_id=None, iata=None).exclude(svg_logo='')
        company_list = [get_company(c) for c in company_model_list]

        update_colors(company_list, options['force'])

        if options['render_html']:
            render_html(company_list)


def get_company(company_model):
    logo_url = 'http://yastatic.net/rasp/media/' + str(company_model.svg_logo)
    logo_file = urllib2.urlopen(logo_url)
    color = get_color(logo_file)

    return {
        'model': company_model,
        'logo_url': logo_url,
        'color': color
    }


def get_color(logo_file):
    image_data = get_image_data(logo_file)
    pixel = get_border_pixel(image_data)

    return rgb2hex(pixel)


def get_border_pixel(image_data):
    pixels = []
    for x in range(0, LOGO_SIZE, LOGO_SIZE - 1):
        for y in range(LOGO_SIZE):
            pixels.append(image_data.getpixel((x, y)))

    for y in range(0, LOGO_SIZE, LOGO_SIZE - 1):
        for x in range(1, LOGO_SIZE - 1):
            pixels.append(image_data.getpixel((x, y)))

    return Counter(pixels).most_common(1)[0][0]


def get_image_data(logo_file):
    raster = cairo.ImageSurface(cairo.FORMAT_ARGB32, LOGO_SIZE, LOGO_SIZE)
    temp_png = tempfile.NamedTemporaryFile()
    ctx = cairo.Context(raster)
    handle = rsvg.Handle(None, str(logo_file.read()))

    handle.render_cairo(ctx)
    raster.write_to_png(temp_png)
    temp_png.seek(0)

    image = Image.open(temp_png)
    image_data = image.getdata()

    temp_png.close()

    return image_data


def check_transparent(rgba):
    return len(rgba) == 4 and rgba[3] == 0


def rgb2hex(rgba):
    rgb = TRANSPARENT_COLOR if check_transparent(rgba) else (rgba[i] for i in range(3))
    return '#' + struct.pack('BBB', *rgb).encode('hex')


def update_colors(company_list, force_update):
    for company in company_list:
        if force_update or company['model'].logo_bgcolor == '':
            update_company_color(company)


def update_company_color(company):
    company['model'].logo_bgcolor = company['color']
    company['model'].save()
    log.info('the color of the company {id} was changed to {color}'.format(id=company['model'].id, color=company['color']))


def render_html(company_list):
    with open(HTML_PATH, 'w') as f:
        f.write("""
            <!DOCTYPE html>
            <html>
                <head>
                    <meta charset="utf-8">
                    <title>Logo SVGs</title>
                    <style>
                        body {{
                            display: flex;
                            flex-wrap: wrap;
                            background-color: #ece8dd;
                        }}
                        div {{
                            margin: 2px;
                            width: 100px;
                        }}
                        div > div {{
                            width: {logo_size}px;
                            height: {logo_size}px;
                            margin: 0;
                            background-size: contain;
                        }}
                    </style>
                </head>
                <body>
        """.format(logo_size=str(LOGO_SIZE)))

        for company in company_list:
            f.write("""
                <div style="background-color: {color}" data-id="{company_id}">
                    <div style="background-image: url({logo_url})"></div>
                </div>
            """.format(color=company['color'], company_id=company['model'].id, logo_url=company['logo_url']))

        f.write("""
                </body>
            </html>
        """)

        log.info('HTML file with logos: {html_path} created'.format(html_path=HTML_PATH))
