# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging
import ujson as json

log = logging.getLogger(__name__)
access = logging.getLogger('access.access')
error = logging.getLogger('access.error')
timeline = logging.getLogger('access.timeline')
slow = logging.getLogger('access.slow')
so_slow = logging.getLogger('access.so_slow')

SLOW_THRESHOLD = 0.5  # в секундах
SO_SLOW_THRESHOLD = 1  # в секундах

DELIMETER = u' $$$$$ '


def _get_curl(request):
    headers = dict(request.headers)

    if 'Accept-Encoding' in headers:
        headers.pop('Accept-Encoding')  # иначе не удобно дергать курлом, так как обычно просят зазипованный контент.

    headers = u' '.join([u'-H "{}: {}"'.format(x, y) for x, y in headers.items()])

    return u'curl "{}" -X POST -d \'{}\' {}'.format(request.url, request.data.decode('utf-8'), headers)


def _get_session_id(request):
    return request.args.get('aviaSessionId', 'UNKNOWN-SESSION')


def _get_source(request):
    return dict(request.headers).get('User-Agent', 'UNKNOWN')


def _get_method(request):
    return request.args.get('name', 'UNKNOWN')


def _try_find_errors(data):
    if isinstance(data, dict) and data.get('status') == 'error':
        return [{'reason': data.get('reason', 'Unknown error')}]
    data = data.get('data')

    errors = []
    if (isinstance(data, list)):
        for d in data:
            if isinstance(d, dict) and d.get('status') == 'error':
                errors.append({'reason': d.get('reason'), 'description': d.get('description')})

    return errors


def _build_message(request, params=[]):
    return DELIMETER + DELIMETER.join([
        _get_source(request) + u'',
        _get_method(request) + u'',
        _get_session_id(request) + u'',
        _get_curl(request)
    ] + params)


def access_jsonrpc_log(request):
    try:
        access.info(_build_message(request))
    except Exception:
        log.exception('Can\'t log access:')


def error_jsonrpc_log(request, response_data):
    try:
        errors = _try_find_errors(response_data)
        if not errors:
            return

        error.info(_build_message(request, [json.dumps(errors)]))
    except Exception:
        log.exception('Can\'t log errors:')


def timeline_jsonrpc_log(request, delta):
    try:
        message = _build_message(request, [str(delta)])
        timeline.info(message)

        if SLOW_THRESHOLD <= delta < SO_SLOW_THRESHOLD:
            slow.info(message)
        if delta >= SO_SLOW_THRESHOLD:
            so_slow.info(message)
    except Exception:
        log.exception('Can\'t log timeline')
