# -*- coding: utf-8 -*-
from __future__ import absolute_import

import os
from collections import namedtuple

from cachetools.func import lru_cache
from django.utils.encoding import force_text, force_bytes
from library.python import resource

from travel.avia.library.python.common.utils.unicode_csv import unicode_reader


RESTRICTIONS_PATH = os.path.join('resfs/file', os.path.dirname(__file__), 'data', 'covid_restrictions.csv')
Restriction = namedtuple('Restriction', field_names=(
    'regionId', 'countryId', 'regionName', 'mask', 'gloves', 'passes', 'selfIsolation', 'borders', 'untilDate',
    'fullText', 'isClosed', 'closedText',
))


@lru_cache(maxsize=None)
def _cache():
    """

    :rtype: typing.Tuple[typing.Dict[int,Restriction],typing.Dict[int,Restriction]]
    """
    region_restrictions = {}
    country_restrictions = {}
    data = [force_bytes(line.strip()) for line in force_text(resource.find(RESTRICTIONS_PATH)).strip().split('\n')]
    reader = unicode_reader(data, encoding='utf-8', delimiter=';')
    title = next(reader)  # noqa

    def _serialize(restriction):
        return {
            'mask': restriction.mask,
            'gloves': restriction.gloves,
            'passes': restriction.passes,
            'selfIsolation': restriction.selfIsolation,
            'borders': restriction.borders,
            'untilDate': restriction.untilDate,
            'fullText': restriction.fullText,
            'isClosed': bool(restriction.isClosed == 'true'),
            'closedText': restriction.closedText,
        }

    for row in reader:
        restriction = Restriction(*row)
        if restriction.regionId:
            region_restrictions[int(restriction.regionId)] = _serialize(restriction)
        elif restriction.countryId:
            country_restrictions[int(restriction.countryId)] = _serialize(restriction)

    return region_restrictions, country_restrictions


def get_region_restrictions(region_id):
    region_restrictions, _ = _cache()
    return region_restrictions.get(region_id, {})


def get_all_country_restrictions():
    _, country_restrictions = _cache()
    return country_restrictions


def get_country_restrictions(country_id):
    return get_all_country_restrictions().get(country_id, {})
