# -*- coding: utf-8 -*-
from __future__ import absolute_import

import flask
import logging
from datetime import datetime, timedelta

from travel.avia.library.python.avia_data.models import MinPrice
from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Station, Settlement

from travel.avia.backend.main.lib.prices import DirectionBasePrice, DirectionPrice, PrefetchedDirectionPrice, get_min_prices_by_keys


log = logging.getLogger(__name__)


class BaseDirectionFinder(object):
    """ Абстрактный класс поиска предложений для города в зависимости от дат и пользователя
        Требует переопределения метода _find (см. recipes)
    """

    TWOWAY_RANGE_MIN = 5
    TWOWAY_RANGE_MAX = 31

    def __init__(self, city_from, when=None, return_date=None, limit=6, pad=4):

        if isinstance(city_from, Station):
            city_from = city_from.settlement

        self.city_from = city_from
        self.when = when
        self.return_date = return_date

        self.limit = limit
        self.pad = pad

        self.national_version = flask.g.get('national_version')

        self.left_border, self.right_border = self._get_borders()

    def find(self):
        """ Поиск всех направлений подходящих под фильтры """

        raise NotImplementedError

    def get_direction(self, city_to, city_from=None):
        """ Поиск одного конкретного направления """

        raise NotImplementedError

    def _get_borders(self):
        """ Если даты не переданы или невалидны ищем за 30 дней """

        city_local_datetime = self.city_from.get_local_datetime(datetime.now())

        if self.when and self.when >= city_local_datetime.date():
            left_date_border = self.when
        else:
            left_date_border = (city_local_datetime + timedelta(days=1)).date()

        if self.return_date and self.return_date >= left_date_border:
            right_date_border = self.return_date
        else:
            right_date_border = (left_date_border + timedelta(days=30))

        return left_date_border, right_date_border

    def _memcache_update(self, directions):
        cache_keys = {}
        for direction in directions:
            cache_keys[direction.price_key(direct=True)] = direction.update_direct
            cache_keys[direction.price_key(direct=False)] = direction.update_indirect

        for key, cached_price in get_min_prices_by_keys(cache_keys.keys()).iteritems():
            cache_keys.get(key)(cached_price)

        return directions


class DirectionRawSQLFinder(BaseDirectionFinder):
    """ Работает так же как DirectionFinder,
        но вытаскивает данные из базы при помощи raw sql
    """

    def get_direction(self, city_to, city_from=None):
        """ Позволяет получать отдельное направление учитывая общие фильтры
            Работает не оптимально!!! По возможности не использовать.
        """

        city_from = city_from or self.city_from

        directions = self._get_directions(city_from, [city_to.id])
        directions = self._memcache_update(directions)

        if directions:
            return directions[0]

        return DirectionBasePrice(city_from, city_to, national_version=self.national_version)

    def _min_prices_to_dict(self, min_prices):
        result = {}

        for m in min_prices:
            key = "%d_%d" % (m.departure_settlement_id, m.arrival_settlement_id)
            result[key] = m

        return result

    def _get_date_where(self):
        if not self.when and not self.return_date:
            return """
                (
                    date_forward BETWEEN '%s' AND '%s' AND
                    (
                        (
                            date_backward >= (date_forward + INTERVAL '%d 0:0:0:0' DAY_MICROSECOND) AND
                            date_backward <= (date_forward + INTERVAL '%d 0:0:0:0' DAY_MICROSECOND)
                        )
                        OR date_backward IS NULL
                    )
                )
            """ % (
                self.left_border.isoformat(), self.right_border.isoformat(),
                self.TWOWAY_RANGE_MIN, self.TWOWAY_RANGE_MAX
            )

        sql = "date_forward = '%s'" % self.when.isoformat()
        if self.return_date:
            sql += " AND date_backward = '%s'" % self.return_date.isoformat()
        else:
            sql += " AND date_backward IS NULL"

        return sql

    def _get_directions(self, city_from, arrival_ids):
        if not arrival_ids:
            return []

        date_where = self._get_date_where()

        # 1. Вытащим города с минимальными ценами подходящие под фильтры.
        #    В результате будет связка откуда+куда+цена
        inner_sql = """
SELECT
departure_settlement_id, arrival_settlement_id, national_version, passengers,
min(price) AS minprice
FROM www_minprice
WHERE
passengers='1_0_0' AND
national_version=%%s AND
%s AND
departure_settlement_id=%%s AND
arrival_settlement_id in (%s)
group by arrival_settlement_id
order by minprice
        """ % (date_where, ','.join([str(i) for i in arrival_ids]))
        # 2. По связке вытаскиваем все записи где подходящие под откуда+куда+цена
        #    группированные по датам. Обычно их получается не много (сотни)
        grouped_sql = """
SELECT m.*
FROM (%s) as x INNER JOIN www_minprice AS m ON
m.departure_settlement_id = x.departure_settlement_id AND
m.arrival_settlement_id = x.arrival_settlement_id AND
m.national_version = x.national_version AND
m.passengers = x.passengers AND
m.price = x.minprice AND
%s
ORDER BY m.date_forward, m.date_backward
        """ % (
            inner_sql,
            # снова нужен where, т.к. откуда+куда+цена могут быть на другие даты
            date_where.replace(
                'date_forward', 'm.date_forward'
            ).replace(
                'date_backward', 'm.date_backward'
            )
        )

        # 3. Вытаскиваем запись с самой ранней датой через subquery (это хак)
        raw_sql = """
select u.* from (%s) as u
group by u.departure_settlement_id, u.arrival_settlement_id, u.price
order by u.price;
        """ % grouped_sql

        min_prices_part1 = list(MinPrice.objects.raw(raw_sql, [
            self.national_version,
            city_from.id
        ]))

        # Какой-нибудь Сидней или США, где вообще нет цен
        if not min_prices_part1:
            return []

        # Достаем прямые/пересадочные цены на эти же даты
        raw_sql2 = []
        for i in min_prices_part1:
            raw_sql2.append("""
SELECT * FROM www_minprice
WHERE
passengers='1_0_0' AND
national_version='%s' AND
date_forward='%s' AND
date_backward%s AND
departure_settlement_id=%d AND
arrival_settlement_id=%d AND
direct_flight=%d
            """ % (
                self.national_version,
                i.date_forward.isoformat(),
                "='%s'" % i.date_backward.isoformat() if i.date_backward else ' IS NULL',
                city_from.id,
                i.arrival_settlement_id,
                0 if i.direct_flight else 1
            ))

        min_prices_part2 = MinPrice.objects.raw(' UNION ALL '.join(raw_sql2) + ';')
        min_prices_part2_dict = self._min_prices_to_dict(min_prices_part2)

        currency_ids = [p.currency_id for p in min_prices_part1]
        currency_ids += [p.currency_id for p in min_prices_part2_dict.values()]
        currencies = {c.id: c for c in list(Currency.objects.filter(id__in=set(currency_ids)))}

        settlement_ids = set([p.arrival_settlement_id for p in min_prices_part1])
        settlements = {
            s.id: s for s in list(Settlement.objects.filter(id__in=settlement_ids))
        }

        # Создаём объекты направлений с обоими ценами
        directions = []
        for min_price in min_prices_part1:
            key = "%d_%d" % (min_price.departure_settlement_id, min_price.arrival_settlement_id)
            min_price2 = min_prices_part2_dict.get(key)
            min_price.currency = currencies[min_price.currency_id]
            if min_price2:
                min_price2.currency = currencies[min_price2.currency_id]

            direction = PrefetchedDirectionPrice(
                city_from, settlements[min_price.arrival_settlement_id],
                min_price=min_price, min_price2=min_price2,
                national_version=self.national_version,
                allow_roughly=True
            )

            directions.append(direction)

        return directions


class SettlementDirectionFinder(DirectionRawSQLFinder):
    def __init__(self, *args, **kwargs):
        self.arrival_ids = kwargs.pop('arrival_ids')

        super(SettlementDirectionFinder, self).__init__(*args, **kwargs)

    def _find(self):
        city_from = self.city_from

        directions = self._get_directions(city_from, self.arrival_ids)

        return directions[:self.limit]

    def find(self):
        directions = self._find()

        found_ids = set([d.city_to.id for d in directions])
        not_founded_ids = [s_id for s_id in self.arrival_ids if s_id not in found_ids]
        no_price_directions = [DirectionBasePrice(
            self.city_from, s,
            date_forward=self.when, date_backward=self.return_date
        ) for s in Settlement.objects.filter(id__in=set(not_founded_ids))]

        directions = self._memcache_update(directions)

        return directions + no_price_directions


class CountryDirectionFinder(DirectionRawSQLFinder):
    def __init__(self, *args, **kwargs):
        self.arrival_ids = kwargs.pop('arrival_ids')

        super(CountryDirectionFinder, self).__init__(*args, **kwargs)

    def _find(self):
        city_from = self.city_from

        directions = self._get_directions(city_from, self.arrival_ids)

        return directions[:self.limit]

    def find(self):
        return self._find()

    def _get_directions(self, city_from, arrival_ids):
        if not arrival_ids:
            return []

        date_where = self._get_date_where()
        national_version = self.national_version

        inner_sql = """
        SELECT *
        FROM www_minprice
        WHERE
        passengers='1_0_0' AND
        national_version=%%s AND
        departure_settlement_id=%%s AND
        arrival_settlement_id in (%s) AND
        %s
        GROUP BY arrival_settlement_id
        ORDER BY price ASC, date_forward, date_backward
                """ % (','.join([str(i) for i in arrival_ids]), date_where)

        raw_sql = """
          SELECT u.* from (%s) as u JOIN www_settlement as s ON (
            s.id = u.arrival_settlement_id
          )
          GROUP BY u.departure_settlement_id, s.country_id, u.arrival_settlement_id, u.price
          ORDER BY u.price;
        """ % inner_sql

        min_prices = list(MinPrice.objects.raw(raw_sql, [
            national_version,
            city_from.id,
        ]))

        return min_prices


def apply_passengers(recipe_options, directions):
    if not directions:
        return []

    passengers = '_'.join([
        str(recipe_options.get('adult_seats', 1)),
        str(recipe_options.get('children_seats', 0)),
        str(recipe_options.get('infant_seats', 0)),
    ])

    if passengers == '1_0_0':
        return directions

    def apply(d):
        if isinstance(d, DirectionPrice):
            d = DirectionPrice(
                city_from=d.city_from,
                city_to=d.city_to,
                national_version=d.national_version,
                date_forward=d.date_forward,
                date_backward=d.date_backward,
                passengers=passengers
            )

        elif isinstance(d, PrefetchedDirectionPrice):
            d.apply_passengers(passengers)

        elif isinstance(d, DirectionBasePrice):
            d.set_passengers(passengers)

        return d

    return [apply(d) for d in directions]
