from __future__ import absolute_import

from travel.avia.backend.repository.station import StationRepository, station_repository  # noqa
from travel.avia.backend.repository.s2s import GeoRelationsRepository, geo_relations_repository  # noqa
from travel.avia.backend.repository.settlement import SettlementRepository, settlement_repository  # noqa


class DisplayCodeManager(object):
    def __init__(self, settlement_repository, station_repository,
                 geo_relation_repository):
        # type: (SettlementRepository, StationRepository, GeoRelationsRepository) -> None
        self._settlement_repository = settlement_repository
        self._airport_repository = station_repository

        self._geo_relation_repository = geo_relation_repository

        self._airport_index = {}
        self._settlement_iata_index = {}
        self._settlement_code_index = {}

    def pre_cache(self):
        airport_index = {}
        for model in self._airport_repository.get_all():
            airport_index[model.pk] = model.iata or model.sirena

        settlement_iata_index = {}
        settlement_code_index = {}
        for model in self._settlement_repository.get_all():
            settlement_iata_index[model.pk] = self._calc_iata_for_settlement(model)
            settlement_code_index[model.pk] = (
                settlement_iata_index.get(model.pk) or
                self._calc_sirena_for_settlement(model)
            )

        self._airport_index = airport_index
        self._settlement_iata_index = settlement_iata_index
        self._settlement_code_index = settlement_code_index

    def _calc_sirena_for_settlement(self, settlement):
        code = settlement.sirena
        if code:
            return code

        airports = [
            self._airport_repository.get(airport_id)
            for airport_id in
            self._geo_relation_repository.get_airport_ids_for(
                settlement.pk
            )
        ]
        airports = filter(None, airports)

        for airport in airports:
            if airport.sirena:
                return airport.sirena

        return None

    def _calc_iata_for_settlement(self, settlement):
        code = settlement.iata
        if code:
            return code

        airports = [
            self._airport_repository.get(airport_id)
            for airport_id in
            self._geo_relation_repository.get_airport_ids_for(
                settlement.pk
            )
        ]
        airports = filter(None, airports)

        for airport in airports:
            if airport.iata:
                return airport.iata

        return None

    def get_code_for_settlement(self, settlement_id):
        return self._settlement_code_index.get(settlement_id)

    def get_iata_for_settlement(self, settlement_id):
        return self._settlement_iata_index.get(settlement_id)

    def get_code_for_airport(self, airport_id):
        return self._airport_index.get(airport_id)


display_code_manager = DisplayCodeManager(
    settlement_repository=settlement_repository,
    station_repository=station_repository,
    geo_relation_repository=geo_relations_repository
)
