# -*- encoding: utf-8 -*-
from __future__ import absolute_import

from cachetools import cached, TTLCache
from cachetools.func import lru_cache
from django.conf import settings
from django.utils.translation import get_language
from django.core.exceptions import ObjectDoesNotExist

from travel.avia.library.python.common.models.geo import Settlement, Station2Settlement, \
    Station, Country
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.models_utils.geo import Point
from travel.avia.library.python.common.views.settlement import get_biggest_cities
from travel.avia.library.python.common.utils.caching import cache_until_switch


def get_model_query_set(model):
    objects = model.objects
    return objects._queryset_class(model=objects.model, using=objects._db, hints=objects._hints)


@cache_until_switch
def has_own_airport(city):
    qs = Station.objects.filter(hidden=False, majority__id__lt=3, t_type=TransportType.PLANE_ID)

    if qs.filter(settlement=city).count():
        return True

    return False


@cache_until_switch
def has_s2s_airport(city):
    qs = Station.objects.filter(hidden=False, majority__id__lt=3, t_type=TransportType.PLANE_ID)

    if qs.filter(station2settlement__settlement=city).count():
        return True

    return False


def has_airport(city):
    return has_own_airport(city) or has_s2s_airport(city)


@cache_until_switch
def get_airport_city(city, national_version):
    # Попробуем найти соответствие в базе
    relation = Station2Settlement.objects.filter(
        settlement=city,
        station__t_type=TransportType.PLANE_ID
    )

    if relation:
        relation = relation[0]  # Не паримся и берём первый попавшийся
        return relation.station.settlement

    # Не получилось - попробуем через столицу региона
    if city.region:
        capital = city.region.get_capital()

        # Учитываем флаг "Не показывать нигде"
        if capital and not capital.hidden and capital != city:
            return capital

    # Есть несколько городов без авиасообщения которые не ложатся ни на какую логику
    if city.id in settings.SETTLEMENT_ALIASES:
        return Settlement.objects.get(id=settings.SETTLEMENT_ALIASES[city.id])

    # Не получилось - Отдадим столицу страны или Москву
    if city.country:
        capital = get_capital(city.country, national_version)
        if capital:
            return capital

    return Settlement.get_default_city()


def get_settlement_by_geoid(geo_id, national_version):
    if geo_id:
        try:
            return Settlement.objects.get(_geo_id=geo_id)
        except Settlement.DoesNotExist:
            pass

    country_id = settings.NATIONAL_VERSION_COUNTRY_MAP[national_version]

    # Хак для турции, отдаём Стамбул
    if country_id == 983:
        return Settlement.objects.get(id=11508)

    return Country.objects.get(id=country_id).get_capital()


@lru_cache(maxsize=None)
def get_point_by_key(key):
    try:
        prefix, geo_id = key[0], key[1:]
    except (IndexError, TypeError):
        raise ValueError("Unknown key %r" % key)

    if prefix == 'g':
        try:
            return Settlement.objects.get(_geo_id=geo_id)
        except Settlement.DoesNotExist:
            pass

        try:
            return Country.objects.get(_geo_id=geo_id)
        except Country.DoesNotExist:
            pass

        raise ObjectDoesNotExist(u'Geo Object with geo_id == %s does not exist' % geo_id)

    model, pk = Point.parse_key(key)

    return model.objects.get(pk=pk)


def phrase_from(geo):
    lang = get_language()

    if lang in ('tr', 'en', 'de'):
        return geo.L_title()

    title = geo.L_title(case='genitive', fallback=False)

    if not title:
        return geo.L_title()

    return title


def phrase_to(geo):
    lang = get_language()

    if lang in ('tr', 'en', 'de'):
        return geo.L_title()

    title = geo.L_title(case='accusative', fallback=False)

    if not title:
        return geo.L_title()

    return title


def get_settlement_by_point(point):
    if isinstance(point, Station):
        try:
            point = point.settlement
        except:
            return None
    elif not isinstance(point, Settlement):
        return None

    # проверим есть ли в городе аэропорт
    if has_own_airport(point):
        return point

    # Нужно замаппить привязанный город в город аэропорта
    relation = Station2Settlement.objects.filter(
        settlement=point,
        station__t_type=TransportType.PLANE_ID
    )

    if relation:
        relation = relation[0]  # Не паримся и берём первый попавшийся
        return relation.station.settlement

    return point


@cached(cache=TTLCache(maxsize=float('inf'), ttl=30 * 60), key=lambda c, national_version: str(c.id) + national_version)
def get_capital(country, national_version):
    # Хак для турции, у неё в базе нет столицы
    # но отдавать надо Стамбул
    if country.id == settings.TURKEY_GEO_ID:
        return Settlement.objects.get(id=settings.ISTANBUL_GEO_ID)

    return country.get_capital()


@cache_until_switch
def get_country_cities(country, amount=settings.CITIES_IN_COUNTRY_SEARCH):
    result = []
    for city, airports in get_biggest_cities(country, amount=amount):
        result.append(city)

    return result
