from __future__ import absolute_import

import heapq
import logging
from itertools import product
from operator import itemgetter
from datetime import timedelta

import ujson
import requests
from django.conf import settings

from travel.avia.library.python.common.utils.iterrecipes import group_by

log = logging.getLogger(__name__)


def search(holidays, from_city, national_version, plus_minus=0, batch_size=2):
    log.info(
        'Start search: [%r] [%r] [%r] [%r] [%r]',
        holidays, from_city, national_version, plus_minus, batch_size
    )
    directions_product = [
        (holiday, holiday_direction)
        for holiday in holidays
        for holiday_direction in holiday.holiday_directions
        if holiday_direction.from_settlement_pk == from_city.id
    ]
    direction_requests = [
        {
            'from_id': holiday_direction.from_settlement_pk,
            'to_id': holiday_direction.to_settlement_pk,
            'forward_date': forward_date.isoformat(),
            'backward_date': backward_date.isoformat()
        }
        for holiday, holiday_direction in directions_product
        for forward_date, backward_date in _two_date_product(
            holiday.start_day, holiday.end_day, plus_minus
        )
    ]

    empty_result = {
        holiday.pk: [] for holiday in holidays
    }
    if not direction_requests:
        log.warn('Empty the direction requests list')
        return empty_result

    log.info(
        'Request the direction requests list [%r] [%r] [%r]',
        settings.AVIA_PRICE_INDEX_URL,
        national_version,
        ujson.dumps({'min_requests': direction_requests})
    )
    raw_response = requests.post(
        '{}/search_methods/v1/min_price_batch_search/{}'.format(settings.AVIA_PRICE_INDEX_URL, national_version),
        json={'min_requests': direction_requests},
        timeout=5
    )
    log.warn('Request before process: [%r]', raw_response.text)
    if raw_response.status_code != 200:
        log.warn('Price index is returned %r', raw_response.status_code)
        return empty_result

    response = raw_response.json()
    if response['status'] != 'ok':
        log.warn('Price index is returned broken status%r', response['status'])
        return empty_result

    records = [
        (holiday.pk, r)
        for r, (holiday, holiday_direction) in zip(response['data'], directions_product)
    ]

    for holiday_id, priced_variants in group_by(records, itemgetter(0)):
        empty_result[holiday_id] = _top_n(priced_variants, batch_size)

    log.warn('Request after process: [%r]', empty_result)

    return empty_result


def _two_date_product(date1, date2, window):
    def neighbors(date):
        return [date + timedelta(days=delta) for delta in range(-window, window + 1)]

    return [
        (d1, d2)
        for d1, d2 in product(neighbors(date1), neighbors(date2))
        if d1 < d2
    ]


def _top_n(priced_variants, number):
    return heapq.nsmallest(
        number,
        [priced_variant for (_, priced_variant) in priced_variants if priced_variant['min_price']],
        key=lambda rec: rec['min_price']['value']
    )
