# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging
import re
import ujson as json
from contextlib import contextmanager

from django.conf import settings

from travel.avia.library.python.avia_data.models import AvgPrice, MinPrice
from travel.avia.library.python.common.models.geo import Settlement, Station
from travel.avia.library.python.common.models.currency import Price

from travel.avia.backend.main.lib.caches import shared_cache
from travel.avia.backend.repository.currency import currency_repository

log = logging.getLogger(__name__)


def key_by_params(
    city_from_key,
    city_to_key,
    national_version,
    passengers_key,
    date_forward,
    date_backward,
    is_direct,
):
    return 'min_price_json_%s_%s_%s_%s_%s_%s_%s' % (
        city_from_key,
        city_to_key,
        national_version,
        passengers_key,
        date_forward.strftime('%Y-%m-%d'),
        date_backward and date_backward.strftime('%Y-%m-%d') or None,
        is_direct and 'direct' or 'transfers',
    )


def get_min_prices_by_keys(keys):
    price_by_key = shared_cache.get_many(keys)

    for key in price_by_key:
        price_by_key[key] = json.loads(price_by_key[key])

    return price_by_key


def get_min_price_by_key(key):
    price_by_key = get_min_prices_by_keys([key])
    return next(iter(price_by_key.values()), None)


def convert_to_national(tariff, national_version, rates):
    try:
        to_currency = settings.AVIA_NATIONAL_CURRENCIES[national_version]
    except KeyError:
        raise ValueError('Not alowed national_version [%s]',
                         national_version)

    if tariff.currency == to_currency:
        return tariff

    if not rates:
        raise ValueError('No rates')

    if tariff.currency not in rates:
        raise ValueError('Tariff currency [%s] not in rates',
                         tariff.currency)

    if to_currency not in rates:
        raise ValueError('Target currency [%s] not in rates',
                         to_currency)

    value = float(tariff.value)
    from_rates = float(rates[tariff.currency])
    to_rates = float(rates[to_currency])
    return value * from_rates / to_rates


class AviaPrice(Price):
    def __init__(self, value, currency=None, iso_currency=None, roughly=False):
        super(AviaPrice, self).__init__(value, currency, iso_currency)

        self.roughly = roughly

    def __json__(self):
        v = super(AviaPrice, self).__json__()

        if self.roughly:
            v['roughly'] = True

        return v


class DirectionBasePrice(object):
    def __init__(self, city_from, city_to, min_price=None, national_version='ru',
                 date_forward=None, date_backward=None, passengers=None, klass=None):
        self.city_from = city_from
        self.city_to = city_to
        self.national_version = national_version

        self.min_price = None
        self._min_price_direct = None
        self._min_price_indirect = None

        if min_price:
            self._save_min_price(min_price)

        self._date_forward = date_forward or getattr(min_price, 'date_forward', None)
        self._date_backward = date_backward or getattr(min_price, 'date_backward', None)

        self._passengers = passengers or getattr(min_price, 'passengers', None) or '1_0_0'
        self._klass = klass or 'economy'

    @property
    def date_forward(self):
        return self._date_forward

    @property
    def date_backward(self):
        return self._date_backward

    @property
    def direct_flight(self):
        return getattr(self.min_price, 'direct_flight', None)

    @property
    def passengers(self):
        return self._passengers

    @property
    def currency(self):
        if self.min_price:
            return self.min_price.currency.code

        return settings.AVIA_NATIONAL_CURRENCIES.get(
            self.national_version,
            settings.AVIA_NATIONAL_CURRENCIES['ru']
        )

    @property
    def price(self):
        return self.min_price

    @property
    def direct_price(self):
        return self._min_price_direct

    @property
    def indirect_price(self):
        return self._min_price_indirect

    def _save_min_price(self, min_price):
        if not self.min_price or self.min_price > min_price:
            self.min_price = min_price

        if min_price.direct_flight:
            self._min_price_direct = min_price
        else:
            self._min_price_indirect = min_price

    def price_key(self, direct, passengers=None):
        return key_by_params(
            city_from_key=self.city_from.point_key,
            city_to_key=self.city_to.point_key,
            national_version=self.national_version,
            passengers_key=passengers or self.passengers,
            date_forward=self.date_forward,
            date_backward=self.date_backward,
            is_direct=direct
        )

    def set_passengers(self, passengers):
        self._passengers = passengers


class DirectionPrice(DirectionBasePrice):
    """ Направление с ценами.

        Имеет методы для просто цены, а так же прямой и пересадочной.
        Актуализирует цену из кэша.

        Конструктор может дополнить направление из MinPrice либо их нужно передать.
    """

    def __init__(self, city_from, city_to, min_price=None, national_version='ru',
                 date_forward=None, date_backward=None, passengers=None, klass=None,
                 allow_roughly=False):
        if not min_price and not date_forward:
            raise Exception(u'Can not create DirectionPrice without price and date_forward')

        self.allow_roughly = allow_roughly

        self.__db_cache = None
        self.__no_db = False

        super(DirectionPrice, self).__init__(city_from, city_to, min_price, national_version,
                                             date_forward, date_backward, passengers, klass)

    @property
    def price(self):
        if self._klass == 'business':
            return None

        if self.min_price:
            return self._cached_price(self.min_price.direct_flight)
        else:
            # Если цена не пришла, то пробуем сначала достать из мэмкэша
            direct_price = self._get_online_price(direct=True)
            indirect_price = self._get_online_price(direct=False)

            if direct_price and not indirect_price:
                return direct_price

            if not direct_price and indirect_price:
                return indirect_price

            if direct_price or indirect_price:
                return min(direct_price, indirect_price)

            # А потом из базы
            return self._get_db_price()

    @property
    def direct_price(self):
        return self._cached_price(direct=True)

    @property
    def indirect_price(self):
        return self._cached_price(direct=False)

    def _cached_price(self, direct):
        name = '__' + re.sub(ur'[^a-zA-Z0-9]', '_',
                             self.price_key(direct))

        if not hasattr(self, name):
            price = self._get_online_price(direct)
            if not price:
                price = self._get_db_price(direct)
            setattr(self, name, price)

        return getattr(self, name)

    def _get_online_price(self, direct):
        if self._klass == 'business':
            return None

        key = self.price_key(direct)
        cached = get_min_price_by_key(key)
        if cached:
            log.debug('Cached min price hit[%s]: %r', key, cached)
            currency = cached['currency']
            iso_currency = currency_repository.get_by_code(currency).iso_code
            return AviaPrice(cached['price'], currency, iso_currency)

        if self.passengers != '1_0_0':
            key = self.price_key(direct, passengers='1_0_0')
            cached = get_min_price_by_key(key)

            if cached:
                currency = cached['currency']
                iso_currency = currency_repository.get_by_code(currency).iso_code
                return AviaPrice(
                    multiply_price(cached['price'], self.passengers),
                    currency,
                    iso_currency,
                    roughly=self.allow_roughly
                )

        return None

    def _get_db_price(self, direct=None):
        """ Получить цены из статистики по направлению """

        if self._klass == 'business':
            return None

        if direct is None and self.min_price:
            return AviaPrice(
                self.min_price.price,
                self.min_price.currency.code,
                self.min_price.currency.iso_code,
                roughly=self.allow_roughly
            )

        if direct and self._min_price_direct:
            return AviaPrice(
                self._min_price_direct.price,
                self._min_price_direct.currency.code,
                self._min_price_direct.currency.iso_code,
                roughly=self.allow_roughly
            )

        if not direct and self._min_price_indirect:
            return AviaPrice(
                self._min_price_indirect.price,
                self._min_price_indirect.currency.code,
                self._min_price_indirect.currency.iso_code,
                roughly=self.allow_roughly
            )

        if self.__no_db:
            return None

        direct_flight_filter = {}

        if direct is not None:
            direct_flight_filter = {'direct_flight': direct}

        def direct_type_filter(mp):
            if direct_flight_filter.get('direct_flight'):
                return mp.direct_flight == direct_flight_filter['direct_flight']

            return True

        if self.__db_cache:
            min_prices = [
                mp for mp in self.__db_cache if (
                    mp.date_forward == self.date_forward and
                    mp.date_backward == self.date_backward and
                    mp.departure_settlement_id == self.city_from.id and
                    mp.arrival_settlement_id == self.city_to.id and
                    mp.passengers == self.passengers and
                    mp.national_version == self.national_version and
                    direct_type_filter(mp)
                )
            ]
        else:
            min_prices = MinPrice.objects.filter(
                national_version=self.national_version,
                departure_settlement_id=self.city_from.id,
                arrival_settlement_id=self.city_to.id,
                date_forward=self.date_forward,
                date_backward=self.date_backward,
                passengers=self.passengers,
                **direct_flight_filter
            ).order_by("price")[:1]

        if min_prices:
            self._save_min_price(min_prices[0])

            return AviaPrice(
                min_prices[0].price,
                min_prices[0].currency.code,
                min_prices[0].currency.iso_code,
                roughly=self.allow_roughly
            )

        if self.passengers != '1_0_0':
            min_prices = MinPrice.objects.filter(
                national_version=self.national_version,
                departure_settlement_id=self.city_from.id,
                arrival_settlement_id=self.city_to.id,
                date_forward=self.date_forward,
                date_backward=self.date_backward,
                passengers='1_0_0',
                **direct_flight_filter
            ).order_by("price")[:1]

            if min_prices:
                self._save_min_price(min_prices[0])

                return AviaPrice(
                    multiply_price(min_prices[0].price, self.passengers),
                    min_prices[0].currency.code,
                    min_prices[0].currency.iso_code,
                    roughly=self.allow_roughly
                )

        return None

    MIN_PRICE_CACHE_TIME = 60 * 90

    @contextmanager
    def set_db_cache(self, db_cache):
        self.__db_cache = db_cache
        try:
            yield
        finally:
            self.__db_cache = None
            self.__no_db = True


class DummyObject(object):
    pass


class PrefetchedDirectionPrice(DirectionBasePrice):
    def __init__(self, city_from, city_to, min_price=None, national_version='ru',
                 date_forward=None, date_backward=None, passengers=None, klass=None,
                 allow_roughly=False, min_price2=None):
        if not min_price:
            raise Exception(u'Can not create PrefetchedDirectionPrice without price')

        self.allow_roughly = allow_roughly
        self._currency_code = min_price.currency.code

        super(PrefetchedDirectionPrice, self).__init__(
            city_from, city_to, min_price, national_version,
            date_forward, date_backward, passengers, klass
        )

        if self._passengers != '1_0_0':
            raise Exception(u'PrefetchedDirectionPrice works only with 1 passenger')

        # Вычисляем цены
        direct_price = None
        indirect_price = None
        price = self._make_avia_price(
            self.min_price.price,
            self.min_price.currency.code,
            self.min_price.currency.iso_code
        )

        if self.min_price.direct_flight:
            direct_price = price
        else:
            indirect_price = price

        if min_price2:
            price = self._make_avia_price(min_price2.price, min_price2.currency.code, min_price2.currency.iso_code)

            if min_price2.direct_flight:
                direct_price = price
            else:
                indirect_price = price

        self.min_prices = {
            'direct': direct_price,
            'indirect': indirect_price,
        }

        # Грязный хак чтобы не сериализовать модель в мемкэш, она большая
        self.min_price = DummyObject()
        setattr(self.min_price, 'date_forward', min_price.date_forward)

    def _make_avia_price(self, price, currency_code, iso_currency_code):
        return AviaPrice(
            multiply_price(price, self.passengers),
            currency_code,
            iso_currency_code,
            roughly=self.allow_roughly
        )

    @property
    def currency(self):
        return self._currency_code

    @property
    def price(self):
        if self._klass == 'business':
            return None

        direct_price = self.min_prices.get('direct')
        indirect_price = self.min_prices.get('indirect')

        if direct_price and not indirect_price:
            return direct_price

        if not direct_price and indirect_price:
            return indirect_price

        if direct_price or indirect_price:
            return min(direct_price, indirect_price)

    @property
    def direct_price(self):
        return self.min_prices.get('direct')

    @property
    def indirect_price(self):
        return self.min_prices.get('indirect')

    def update_direct(self, cached_price):
        currency = cached_price['currency']
        iso_currency = currency_repository.get_by_code(currency).iso_code
        self.min_prices['direct'] = AviaPrice(
            multiply_price(cached_price['price'], self.passengers),
            currency,
            iso_currency,
        )

    def update_indirect(self, cached_price):
        currency = cached_price['currency']
        iso_currency = currency_repository.get_by_code(currency).iso_code
        self.min_prices['indirect'] = AviaPrice(
            multiply_price(cached_price['price'], self.passengers),
            currency,
            iso_currency,
        )

    def _apply_price(self, direction):
        price = self.min_prices[direction]
        if price:
            self.min_prices[direction] = AviaPrice(
                multiply_price(price.value, self.passengers),
                price.currency,
                price.iso_currency,
                price.roughly
            )

    def apply_passengers(self, passengers):
        self._passengers = passengers
        self._apply_price('direct')
        self._apply_price('indirect')


class DummyDirectionPrice(object):
    """
    Класс для хранения цен по направлению.
    direct_price и inderect_price - должны содержать цены для одного пассажира эконом-класса
    """
    def __init__(self, city_from, city_to, direct_price, indirect_price, national_version,
                 date_forward, date_backward, allow_roughly=False):
        assert direct_price or indirect_price is not None, 'Direct or Indirect price must be set'
        if direct_price and indirect_price:
            assert direct_price.currency == indirect_price.currency, 'Prices must have the same currency'

        self._direct_price = direct_price
        self._indirect_price = indirect_price
        self._price = min(direct_price or indirect_price, indirect_price or direct_price)
        self.currency = self._price.currency

        self.city_from = city_from
        self.city_to = city_to

        self.date_forward = date_forward
        self.date_backward = date_backward

        self.passengers_key = '1_0_0'
        self.allow_roughly = allow_roughly

    @property
    def direct_price(self):
        if self._direct_price is None:
            return None
        return AviaPrice(
            value=multiply_price(self._direct_price.value, self.passengers_key),
            currency=self.currency,
            iso_currency=currency_repository.get_by_code(self.currency).iso_code,
            roughly=self.allow_roughly
        )

    @property
    def indirect_price(self):
        if self._indirect_price is None:
            return None
        return AviaPrice(
            value=multiply_price(self._indirect_price.value, self.passengers_key),
            currency=self.currency,
            iso_currency=currency_repository.get_by_code(self.currency).iso_code,
            roughly=self.allow_roughly
        )

    @property
    def price(self):
        direct_price = self.direct_price
        indirect_price = self.indirect_price

        if direct_price and not indirect_price:
            return direct_price

        if not direct_price and indirect_price:
            return indirect_price

        if direct_price or indirect_price:
            return min(direct_price, indirect_price)


def get_point_settlement(point, default=None):
    if isinstance(point, Settlement):
        return point

    if isinstance(point, Station) and point.settlement:
        return point.settlement

    return default


def _get_passenger_coeff(passengers_key):
    if passengers_key == '1_0_0':
        return 1
    adults, children, infants = passengers_key.split('_')
    return int(adults) + int(children) * 0.6 + int(infants) * 0.1


def multiply_price(price, passengers_key):
    return price * _get_passenger_coeff(passengers_key)


def divide_price(price, passengers_key):
    return price / _get_passenger_coeff(passengers_key)


def get_average_price(form, when, return_date, national_version):
    average_price = None

    average_prices = AvgPrice.objects.filter(
        departure_settlement=form.from_point,
        arrival_settlement=form.to_point,
        passengers=form.get_passengers_key(),
        month_forward=when.month,
        year_forward=when.year,
        month_backward=return_date.month if return_date else None,
        year_backward=return_date.year if return_date else None,
        national_version=national_version
    ).order_by('price')[:1]

    if len(average_prices):
        average_price = average_prices[0]

    if not average_price and form.get_passengers_key() != '1_0_0':
        average_prices = AvgPrice.objects.filter(
            departure_settlement=form.from_point,
            arrival_settlement=form.to_point,
            passengers='1_0_0',
            month_forward=when.month,
            year_forward=when.year,
            month_backward=return_date.month if return_date else None,
            year_backward=return_date.year if return_date else None,
            national_version=national_version
        ).order_by('price')[:1]

        if len(average_prices):
            average_price = average_prices[0]
            average_price.price = multiply_price(average_price.price, form.get_passengers_key())

    return average_price


def partner_price(partner, national_version):
    price = partner.get_national_price(national_version)
    return price.get_cents_dispersed()


def rebase_tariff(tariff, national_version, rates):
    try:
        tariff.base_value = convert_to_national(tariff, national_version, rates)
        return tariff
    except ValueError as e:
        log.warning('{national_version} {stack}'.format(national_version=national_version,
                                                        stack=repr(e)))
        return tariff
