# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import logging
from django.conf import settings

from travel.avia.library.python.avia_data.libs.currency.rates.caching import CurrencyRatesCache

from travel.avia.backend.main.lib.caches import default_cache

log = logging.getLogger(__name__)


def fetch_rates_info(national_version):
    rates_cache = CurrencyRatesCache.load(national_version, cache=default_cache)
    if rates_cache:
        rates = rates_cache.rates
    else:
        currency_code = settings.AVIA_NATIONAL_CURRENCIES[national_version]
        rates = {
            currency_code: 1,
        }

    return rates


def convert_tariff(tariff, rates, currency_to):
    if tariff.currency == currency_to:
        return tariff

    conditions = rates and tariff.currency in rates and currency_to in rates

    if conditions:
        tariff.base_value = (
            float(tariff.value) *
            float(rates[tariff.currency]) /
            float(rates[currency_to])
        )
        tariff.value = tariff.base_value
        tariff.currency = currency_to

        return tariff

    if not rates:
        log.warning('rates are not defined')
    elif tariff.currency not in rates:
        log.warning('{currency} is not known'.format(currency=tariff.currency))
    elif currency_to not in rates:
        log.warning('Converting to {currency} which is not in rates'.format(currency=currency_to))

    return None


def get_min_tariff(variants, currency_to):
    rates_cache = CurrencyRatesCache.load(cache=default_cache)
    rates = rates_cache.rates if rates_cache else None

    min_tariff = None

    for v in variants:
        tariff = v.get('tariff') if isinstance(v, dict) else v.tariff

        converted_tariff = convert_tariff(tariff, rates, currency_to)

        if converted_tariff is None:
            continue

        if not min_tariff or min_tariff.value > converted_tariff.value:
            min_tariff = converted_tariff

    return min_tariff
