# -*- coding: utf-8 -*-
from __future__ import absolute_import

from django.conf import settings
from django.utils.text import slugify

from travel.avia.library.python.avia_data.models import MinPrice
from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.common.utils.caching import cached
from travel.avia.backend.main.lib.directions import apply_passengers
from travel.avia.backend.main.lib.prices import PrefetchedDirectionPrice, divide_price
from travel.avia.backend.main.lib.recipes.popular import PopularDirectionFinder


def _min_price_airline_directions_cache_key(finder, airline_slug=None):
    city_from = finder.city_from

    key = 'min_price_airline_directions/%s_%s_%d_%s_%s_%s' % (
        city_from.id, finder.national_version, finder.limit or 0,
        str(finder.left_border), str(finder.right_border),
        airline_slug
    )

    return key


def _normalize_price(min_price):
    """Пересчитываем цену в расчёте на одного пассажира в одну сторону"""
    if min_price.date_backward:
        min_price.price /= 2
        min_price.date_backward = None
    if min_price.passengers != '1_0_0':
        min_price.price = divide_price(min_price.price, min_price.passengers)
        min_price.passengers = '1_0_0'


def _get_min_price_dict_key(min_price):
    return "%d_%d" % (
        min_price.departure_settlement_id,
        min_price.arrival_settlement_id)


def _min_prices_to_dict(min_prices):
    result = {}
    for m in min_prices:
        key = _get_min_price_dict_key(m)
        if not (result.get(key) and result[key].price < m.price):
            result[key] = m
    return result


class AirlineDirectionFinder(PopularDirectionFinder):
    @cached(_min_price_airline_directions_cache_key, timeout=settings.MIN_PRICES_CACHE_TTL)
    def find(self, airline_slug=None):
        arrival_ids = self.get_top_direction_ids(self.city_from, self.national_version, self.limit * 2)
        directions = self._get_directions(arrival_ids, airline_slug)
        directions = sorted(directions, key=lambda x: arrival_ids.index(x.city_to.id))

        return directions[:self.limit]

    def _get_date_where(self):
        if not self.when:
            return "date_forward BETWEEN '%s' AND '%s'" % (
                self.left_border.isoformat(), self.right_border.isoformat()
            )
        return "date_forward = '%s'" % self.when.isoformat()

    def _get_airline_join(self, slug):
        if not slug:
            return ''
        return """
INNER JOIN www_minprice_airlines ON
    www_minprice.id = www_minprice_airlines.minprice_id
INNER JOIN www_company ON
    www_company.id = www_minprice_airlines.company_id AND
    www_company.slug = '%s'
""" % (slugify(slug), )

    def _get_directions(self, arrival_ids, airline_slug=None):
        city_from = self.city_from

        if not arrival_ids:
            return []

        date_where = self._get_date_where()
        airline_join = self._get_airline_join(airline_slug)

        # 1. Вытащим города с минимальными ценами подходящие под фильтры.
        #    В результате будет связка откуда+куда+цена
        inner_sql = """
SELECT
departure_settlement_id, arrival_settlement_id, national_version, passengers,
min(price) AS minprice
FROM www_minprice
%s
WHERE
national_version=%%s AND
%s AND
departure_settlement_id=%%s AND
arrival_settlement_id in (%s)
group by arrival_settlement_id
order by minprice
        """ % (airline_join, date_where, ','.join([str(i) for i in arrival_ids]))
        # 2. По связке вытаскиваем все записи где подходящие под откуда+куда+цена
        #    группированные по датам. Обычно их получается не много (сотни)
        grouped_sql = """
SELECT m.*
FROM (%s) as x INNER JOIN www_minprice AS m ON
m.departure_settlement_id = x.departure_settlement_id AND
m.arrival_settlement_id = x.arrival_settlement_id AND
m.national_version = x.national_version AND
m.passengers = x.passengers AND
m.price = x.minprice AND
%s
ORDER BY m.date_forward, m.date_backward
        """ % (
            inner_sql,
            # снова нужен where, т.к. откуда+куда+цена могут быть на другие даты
            date_where.replace('date_forward', 'm.date_forward')
        )

        # 3. Вытаскиваем запись с самой ранней датой через subquery (это хак)
        raw_sql = """
select u.* from (%s) as u
group by u.departure_settlement_id, u.arrival_settlement_id, u.price
order by u.price;
        """ % grouped_sql

        min_prices_part1 = list(MinPrice.objects.raw(raw_sql, [
            self.national_version,
            city_from.id
        ]))

        # Какой-нибудь Сидней или США, где вообще нет цен
        if not min_prices_part1:
            return []

        # Достаем прямые/пересадочные цены на эти же даты
        raw_sql2 = []
        for i in min_prices_part1:
            raw_sql2.append("""
SELECT * FROM www_minprice
WHERE
passengers='1_0_0' AND
national_version='%s' AND
date_forward='%s' AND
date_backward%s AND
departure_settlement_id=%d AND
arrival_settlement_id=%d AND
direct_flight=%d
            """ % (
                self.national_version,
                i.date_forward.isoformat(),
                "='%s'" % i.date_backward.isoformat() if i.date_backward else ' IS NULL',
                city_from.id,
                i.arrival_settlement_id,
                0 if i.direct_flight else 1
            ))

        min_prices_part2 = MinPrice.objects.raw(' UNION ALL '.join(raw_sql2) + ';')

        for m in min_prices_part1:
            _normalize_price(m)
        for m in min_prices_part2:
            _normalize_price(m)

        min_prices_part1_dict = _min_prices_to_dict(min_prices_part1)
        min_prices_part2_dict = _min_prices_to_dict(min_prices_part2)

        currency_ids = [p.currency_id for p in min_prices_part1]
        currency_ids += [p.currency_id for p in min_prices_part2_dict.values()]
        currencies = {c.id: c for c in list(Currency.objects.filter(id__in=set(currency_ids)))}

        settlement_ids = set([p.arrival_settlement_id for p in min_prices_part1])
        settlements = {s.id: s for s in list(Settlement.objects.filter(id__in=settlement_ids))}

        # Создаём объекты направлений
        directions = []

        for min_price1 in min_prices_part1_dict.itervalues():
            key = _get_min_price_dict_key(min_price1)
            min_price2 = min_prices_part2_dict.get(key)
            min_price1.currency = currencies[min_price1.currency_id]
            if min_price2:
                min_price2.currency = currencies[min_price2.currency_id]

            direction = PrefetchedDirectionPrice(
                city_from, settlements[min_price1.arrival_settlement_id],
                min_price=min_price1, min_price2=min_price2,
                national_version=self.national_version,
                allow_roughly=True
            )

            directions.append(direction)

        return directions


def get_airline_directions(city_from, when, return_date, passengers, extra_ids, limit=None, airline_slug=None):
    directions = AirlineDirectionFinder(city_from, when, return_date, limit).find(airline_slug)

    return apply_passengers(passengers, directions)
