# -*- coding: utf-8 -*-
from __future__ import absolute_import

import logging

import flask
from django.conf import settings
from django.core.cache import cache
from retrying import retry, RetryError
from traceback import format_exc

from travel.avia.library.python.avia_data.models import AviaRecipe, AviaDirectionNational, MinPrice
from travel.avia.library.python.common.models.currency import Currency
from travel.avia.library.python.common.models.geo import Settlement
from travel.avia.library.python.common.utils.caching import cached

from travel.avia.backend.main.lib.directions import DirectionRawSQLFinder, apply_passengers
from travel.avia.backend.main.lib.prices import DirectionBasePrice, PrefetchedDirectionPrice

log = logging.getLogger(__name__)


def min_price_direction_cache_key(finder, city_from, city_to_id):
    key = 'min_price_direction/%s_%s_%s_%s_%s' % (
        city_from.id, city_to_id, finder.national_version,
        str(finder.left_border), str(finder.right_border)
    )

    return key


def min_price_directions_cache_key(finder, city_from=None):
    city_from = city_from or finder.city_from

    key = 'min_price_directions/%s_%s_%d_%s_%s' % (
        city_from.id, finder.national_version, finder.limit,
        str(finder.left_border), str(finder.right_border)
    )

    return key


class PopularDirectionFinder(DirectionRawSQLFinder):
    """ Поиск популярных предложений для города в зависимости от дат и пользователя """

    @staticmethod
    def get_top_direction_ids(city_from, national_version, limit):
        qs = AviaDirectionNational.objects.filter(
            departure_settlement_id=city_from.id,
            national_version=national_version
        )
        # RASPTICKETS-3005 Политические проблемы
        if national_version == 'ua':
            qs = qs.exclude(arrival_settlement__country_id=settings.RUSSIA_GEO_ID)

        arrival_settlements = qs.order_by('-popularity')[:limit]
        return list(arrival_settlements.values_list('arrival_settlement', flat=True))

    @cached(min_price_directions_cache_key, timeout=settings.MIN_PRICES_CACHE_TTL)
    def _find(self, city_from=None):
        city_from = city_from or self.city_from

        arrival_ids = self.get_top_direction_ids(city_from, self.national_version, self.limit * 2)
        directions = self._get_directions(city_from, arrival_ids)

        directions = sorted(
            directions,
            key=lambda x: arrival_ids.index(x.city_to.id)
        )

        return directions[:self.limit]

    def _add_pad_directions(self, directions):
        # Столько городов ещё нужно найти
        pad = self.pad - len(directions)
        already_have_ids = set([d.city_to.id for d in directions])

        capital = None

        pad_direction = []
        pad_settlements = []

        # Маленький город - показываем популярные направления для столицы области
        is_small_city = self.city_from.majority_id > 2 and self.city_from.region
        if is_small_city and not self.city_from.disputed_territory:
            capital = self.city_from.region.get_capital()

            # Учитываем флаг "Не показывать нигде"
            if capital and capital.hidden:
                capital = None

            if capital:
                # Попробуем добавить столичные
                capital_directions = self._find(city_from=capital)

                for direction in capital_directions:
                    if len(pad_direction) >= pad:
                        break

                    if direction.city_to.id in already_have_ids:
                        continue

                    pad_direction.append(direction)
                    already_have_ids.add(direction.city_to.id)

        # Если нет области или всё равно направлений не хватает
        pad = pad - len(pad_direction)

        if pad:
            # Берём большие города страны
            country = (
                self.city_from.translocal_country(self.national_version)
                or flask.g.user_city
            )

            pad_queryset = (
                Settlement.objects
                .filter(country=country)
                .filter(station__t_type__code='plane', station__hidden=False)
                .exclude(id=self.city_from.id)
                .distinct()
                .order_by('?')
            )

            if capital:
                pad_queryset.exclude(id=capital.id)

            pad_settlements = list(
                pad_queryset
                .filter(majority__lt=4)
                .exclude(id__in=already_have_ids)[:pad]
            )

        # Всё равно городов может не хватить, тогда берём даже обычные города
        pad_num = pad - len(pad_settlements)

        if pad_num:
            for settlement in pad_settlements:
                already_have_ids.add(settlement.id)

            pad_settlements += list(
                pad_queryset
                .filter(majority__lt=5)
                .exclude(id__in=already_have_ids)[:pad_num]
            )

        return directions + pad_direction, [
            DirectionBasePrice(capital or self.city_from, settlement)
            for settlement in pad_settlements
        ]

    def _get_cached_directions(self, city_from, arrival_ids):
        keys = [min_price_direction_cache_key(self, city_from, to_id) for to_id in arrival_ids]
        return filter(None, cache.get_many(keys).values())

    def _save_cached_directions(self, directions):
        for d in directions:
            key = min_price_direction_cache_key(self, d.city_from, d.city_to.id)
            cache.set(key, d, settings.MIN_PRICES_CACHE_TTL)

    def _add_extra_ids(self, common_directions, arrival_ids):
        """ Добавление направлений которыми интересовался пользователь """

        common_arrival_ids = set([d.city_to.id for d in common_directions])

        # Список городов которых нет в популярных направлениях
        new_ids = set(arrival_ids) - common_arrival_ids

        if new_ids:
            # проверим мемкэш
            cached_directions = self._get_cached_directions(self.city_from, new_ids)
            new_ids = set(new_ids) - set([d.city_to.id for d in cached_directions])

            # остатки берём из базы и сохраняем в кэш
            db_directions = self._get_directions(self.city_from, new_ids)
            self._save_cached_directions(db_directions)

            new_directions = cached_directions + db_directions
        else:
            new_directions = []

        # Сортируем сначала extra направления,
        # потом попса с оригинальной сортировкой
        extra_from_common_directions = []
        old_common_directions = []
        for d in common_directions:
            if d.city_to.id in arrival_ids:
                extra_from_common_directions.append(d)
            else:
                old_common_directions.append(d)

        extra_directions = new_directions + extra_from_common_directions

        extra_directions = sorted(
            extra_directions,
            key=lambda x: arrival_ids.index(x.city_to.id)
        )

        return (extra_directions + old_common_directions)[:self.limit]

    def find(self, extra_ids=None):
        directions = self._find()
        no_price_directions = []

        if len(directions) < self.pad:
            directions, no_price_directions = self._add_pad_directions(directions)

        if extra_ids:
            directions = self._add_extra_ids(directions, extra_ids)

        directions = directions[:self.limit]
        directions = self._memcache_update(directions)

        return directions + no_price_directions

    def _get_directions(self, city_from, arrival_ids):
        if not arrival_ids:
            return []

        date_where = self._get_date_where()

        # 1. Вытащим города с минимальными ценами подходящие под фильтры.
        #    В результате будет связка откуда+куда+цена
        inner_sql = """
SELECT
departure_settlement_id, arrival_settlement_id, national_version, passengers,
min(price) AS minprice
FROM www_minprice
WHERE
passengers='1_0_0' AND
national_version=%%s AND
%s AND
departure_settlement_id=%%s AND
arrival_settlement_id in (%s)
group by arrival_settlement_id
order by minprice
        """ % (date_where, ','.join([str(i) for i in arrival_ids]))
        # 2. По связке вытаскиваем все записи где подходящие под откуда+куда+цена
        #    группированные по датам. Обычно их получается не много (сотни)
        grouped_sql = """
SELECT m.*
FROM (%s) as x INNER JOIN www_minprice AS m ON
m.departure_settlement_id = x.departure_settlement_id AND
m.arrival_settlement_id = x.arrival_settlement_id AND
m.national_version = x.national_version AND
m.passengers = x.passengers AND
m.price = x.minprice AND
%s
ORDER BY m.date_forward, m.date_backward
        """ % (
            inner_sql,
            # снова нужен where, т.к. откуда+куда+цена могут быть на другие даты
            date_where.replace(
                'date_forward', 'm.date_forward'
            ).replace(
                'date_backward', 'm.date_backward'
            )
        )

        # 3. Вытаскиваем запись с самой ранней датой через subquery (это хак)
        raw_sql = """
select u.* from (%s) as u
group by u.departure_settlement_id, u.arrival_settlement_id, u.price
order by u.price;
        """ % grouped_sql

        try:
            min_prices_part1 = self._raw_sql_with_retry(
                lambda: list(MinPrice.objects.raw(raw_sql, [self.national_version, city_from.id])))
        except RetryError:
            return []

        # Какой-нибудь Сидней или США, где вообще нет цен
        if not min_prices_part1:
            return []

        currency_ids = [p.currency_id for p in min_prices_part1]
        currencies = {c.id: c for c in list(
            Currency.objects.filter(id__in=set(currency_ids)))}

        settlement_ids = set(
            [p.arrival_settlement_id for p in min_prices_part1])
        settlements = {
            s.id: s for s in
            list(Settlement.objects.filter(id__in=settlement_ids))
        }

        # Создаём объекты направлений с обоими ценами
        directions = []
        for min_price in min_prices_part1:
            min_price.currency = currencies[min_price.currency_id]

            direction = PrefetchedDirectionPrice(
                city_from, settlements[min_price.arrival_settlement_id],
                min_price=min_price, min_price2=None,
                national_version=self.national_version,
                allow_roughly=True
            )

            directions.append(direction)

        return directions

    @retry(stop_max_attempt_number=2, retry_on_result=lambda result: result is None)
    def _raw_sql_with_retry(self, run_sql_function):
        try:
            return run_sql_function()
        except:
            log.warning('Exception while calling raw sql: %r', format_exc())
            return None


# Псевдо-рецепт
popular_recipe = AviaRecipe(
    id=0,
    title_ru=u'Популярные направления',  # позже сделать тайтлы ключём, например popular
    title_uk=u'Популярные направления',  # а перевод брать прямо в блоке
    title_tr=u'Популярные направления',
    enabled_ru=True,
    enabled_ua=True,
    enabled_tr=True,
)


def get_popular_directions(city_from, when, return_date, passengers, extra_ids, limit=None):
    directions = PopularDirectionFinder(city_from, when, return_date, limit).find(extra_ids)

    return apply_passengers(passengers, directions)
