from __future__ import absolute_import

from django.conf import settings

from unidecode import unidecode

from travel.avia.backend.main.api_types.settlement import get_iata_code
from travel.avia.backend.main.lib.geo import has_airport


class SettlementSerializer(object):
    def __init__(self):

        self._cache = {}

    def serialize(self, settlement, national_version, lang):
        if not settlement:
            return None

        if settlement.pk not in self._cache:
            self._cache[settlement.pk] = self._serialize(
                settlement, national_version, lang
            )

        return self._cache[settlement.pk]

    def _serialize(self, settlement, national_version, lang):
        if settlement is None:
            return None

        iata = get_iata_code(settlement)
        country = settlement.country
        return {
            'id': settlement.pk,
            'key': settlement.point_key,
            'title': settlement.L_title(),
            'urlTitle': self._get_url_title(settlement, national_version),
            'code': iata or settlement.sirena_id,
            'geoId': settlement._geo_id,
            'iataCode': iata,
            'country': country and {
                "id": country.pk,
                "title": country.L_title(),
                "urlTitle": self._get_url_title(country, national_version)
            },
            'phraseTo': self._get_phrase_to(settlement, lang),
            'phraseFrom': self._get_phrase_from(settlement, lang),
            'hasAirport': has_airport(settlement),
        }

    def _get_url_title(self, s, national_version):
        lang, _ = settings.DOMAIN_LANGUAGE_MAP.get(
            national_version, 'en')
        return unidecode(s.L_title(lang=lang)).replace("'", '')

    def _get_phrase_to(self, s, lang):
        if lang not in ('ru', 'uk'):
            return s.L_title()

        title = s.L_title(case='accusative', fallback=False)

        if not title:
            return s.L_title()

        return title

    def _get_phrase_from(self, s, lang):
        if lang not in ('ru', 'uk'):
            return s.L_title()

        title = s.L_title(case='genitive', fallback=False)

        if not title:
            return s.L_title()

        return title
