# coding=utf-8
from __future__ import absolute_import

from logging import getLogger, Logger  # noqa
from marshmallow import Schema, fields, post_load, ValidationError

from travel.avia.backend.main.rest.airlines.helpers import AbcAirlineProvider, abc_airline_provider  # noqa
from travel.avia.backend.main.rest.helpers import CommonView


class AirlinesForm(Schema):
    national_version = fields.String(required=True)
    lang = fields.String(required=True)
    start = fields.String(required=True)
    end = fields.String(required=True)

    @post_load
    def validate_fields(self, data):
        letters = None
        russian_abc_letters = abc_airline_provider.RUSSIAN_ABC_LETTERS
        abc_letters = abc_airline_provider.ABC_LETTERS

        if data[u'start'] in russian_abc_letters and data[u'end'] in russian_abc_letters:
            letters = russian_abc_letters
        elif data[u'start'] in abc_letters and data[u'end'] in abc_letters:
            letters = abc_letters

        if letters is None:
            message = u'[{start}] or/and [{end}] are not in abcs.'.format(
                start=data[u'start'],
                end=data[u'end']
            )
            raise ValidationError({
                u'start': message,
                u'end': message,
            })

        start = letters.index(data[u'start'])
        end = letters.index(data[u'end'])

        if start > end:
            message = u'Start [{start}] must be less than end [{end}]. Abc: [{abc}]'.format(
                start=data[u'start'],
                end=data[u'end'],
                abc=letters
            )
            raise ValidationError({
                u'start': message,
                u'end': message,
            })

        return dict(data, **{'letters': letters[start:end + 1]})


class AirlinesByAbcView(CommonView):
    def __init__(self, abc_airline_provider, logger):
        # type: (AbcAirlineProvider, Logger) -> None
        super(AirlinesByAbcView, self).__init__(AirlinesForm(), logger)
        self._abc_airline_provider = abc_airline_provider

    def _process(self, parsed_data):
        lang = parsed_data[u'lang']
        airlines_by_letter = self._abc_airline_provider.get_airlines_by_letter(
            lang=lang
        )

        return [
            {
                u'letter': letter,
                u'airlines': [{
                    u'id': c.pk,
                    u'title': c.get_title_with_code(lang),
                    u'slug': c.slug,
                } for c in sorted(
                    airlines_by_letter[letter],
                    key=lambda x: x.get_title_with_code(lang=lang).lower()
                )]
            } for letter in parsed_data[u'letters']
        ]

airlines_by_abc_view = AirlinesByAbcView(
    abc_airline_provider=abc_airline_provider,
    logger=getLogger(__name__)
)
