# -*- coding: utf-8 -*-
from __future__ import absolute_import
from abc import abstractproperty, ABCMeta
from logging import getLogger, Logger  # noqa
from marshmallow import Schema, fields

from travel.avia.backend.main.rest.helpers import CommonView, NotFoundError
from travel.avia.backend.repository.airlines import AirlineModel, AirlineRepository, airline_repository  # noqa
from travel.avia.backend.repository.airline_rating import AirlineRatingRepository, airline_rating_repository  # noqa


class AirlinesInfoSchema(Schema):
    lang = fields.String(required=False, missing='ru')
    fields = fields.Function(
        deserialize=lambda v: set(filter(None, v.split(','))), missing=''
    )


class AirlineInfoSchema(AirlinesInfoSchema):
    company_id = fields.Integer(required=True)


class AbstractAirlineModelView(CommonView):
    __metaclass__ = ABCMeta
    SCHEME = abstractproperty()

    def __init__(self, airline_repository, airline_rating_repository, logger):
        # type: (AirlineRepository, AirlineRatingRepository, Logger) -> None
        self._airline_repository = airline_repository
        self._airline_rating_repository = airline_rating_repository
        super(AbstractAirlineModelView, self).__init__(self.SCHEME, logger)

    def airline_data(self, company_id, fields, lang):
        airline = self._airline_repository.get(company_id)
        if not airline:
            raise NotFoundError('Airline %d not found' % company_id)
        data = {
            'id': self._get_or_none(airline, 'pk'),
            'alliance': self._get_or_none(airline, 'alliance_id'),
            'title': airline.get_title(lang),
            'url': self._get_or_none(airline, 'url'),
            'logoSvg': self._get_or_none(airline, 'logo'),
            'color': self._get_or_none(airline, 'logo_bgcolor'),
            'slug': self._get_or_none(airline, 'slug'),
            'iata': self._get_or_none(airline, 'iata'),
            'sirena': self._get_or_none(airline, 'sirena'),
            'icao': self._get_or_none(airline, 'icao'),
            'rating': self._airline_rating(company_id),
            'default_tariff': self._get_or_none(airline, 'tariff'),
            'seo_description_i18n_key': self._get_or_none(airline,
                                                          'seo_description_key'),
            'baggage_rules': self._get_or_none(airline, 'baggage_rules'),
            'baggage_rules_url': self._get_or_none(airline,
                                                   'baggage_rules_url'),
            'registration_url': airline.get_registration_url(lang),
            'registration_phone': airline.get_registration_phone(lang),
            'hidden': self._get_or_none(airline, 'hidden'),
            'carryon': {
                'length': airline.carryon_length,
                'width': airline.carryon_width,
                'height': airline.carryon_height,
                'dimensions_sum': airline.carryon_dimensions_sum,
            },
            'baggage': {
                'length': airline.baggage_length,
                'width': airline.baggage_width,
                'height': airline.baggage_height,
                'dimensions_sum': airline.baggage_dimensions_sum,
            },
        }

        return {k: data[k] for k in fields} if fields else data

    def _airline_rating(self, company_id):
        # type: (int) -> dict
        return self._airline_rating_repository.get(company_id)

    @staticmethod
    def _get_or_none(airline, attr):
        # type: (AirlineModel, str) -> any
        return getattr(airline, attr, None)


class AirlinesInfoView(AbstractAirlineModelView):
    SCHEME = AirlinesInfoSchema()

    def _process(self, parsed_data):
        # type: (dict) -> dict
        airlines = self._airline_repository.get_all()
        return {
            a.pk: self.airline_data(
                a.pk, parsed_data['fields'], parsed_data['lang']
            ) for a in airlines
        }


class AirlineInfoView(AbstractAirlineModelView):
    SCHEME = AirlineInfoSchema()

    def _process(self, parsed_data):
        # type: (dict) -> dict
        return self.airline_data(
            parsed_data['company_id'], parsed_data['fields'],
            parsed_data['lang']
        )


airlines_info_view = AirlinesInfoView(
    airline_repository, airline_rating_repository, getLogger(__name__)
)

airline_info_view = AirlineInfoView(
    airline_repository, airline_rating_repository, getLogger(__name__)
)
