# -*- coding: utf-8 -*-
from __future__ import absolute_import

from logging import getLogger, Logger  # noqa
from marshmallow import Schema

from django.conf import settings

from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.library.python.common.models.schedule import CompanySynonym
from travel.avia.backend.repository.airlines import airline_repository, AirlineRepository  # noqa


class AirlinesSynonymsView(CommonView):
    def __init__(self, airline_repository, logger):
        # type: (AirlineRepository, Logger) -> None
        super(AirlinesSynonymsView, self).__init__(Schema(), logger)
        self._airline_repository = airline_repository

    def _process(self, parsed_data):
        all_aviacompanies = self._airline_repository.get_all()
        all_synonyms = CompanySynonym.objects.values('company_id', 'synonym', 'language')

        groupped_synonyms = self._group_synonyms(all_synonyms)
        self._add_companies_titles(groupped_synonyms, all_aviacompanies)
        self._remove_empty_entries(groupped_synonyms)

        response = {
            'airlines': [
                {
                    'id': aviacompany.pk,
                    'iata': aviacompany.iata,
                    'sirena': aviacompany.sirena,
                    'icao': aviacompany.icao,
                    'icao_ru': aviacompany.icao_ru,
                    'titles':  groupped_synonyms[aviacompany.pk],
                }
                for aviacompany in all_aviacompanies
            ]
        }

        return response

    def _remove_empty_entries(self, groupped_synonyms):
        for _, translations in groupped_synonyms.iteritems():
            for language, language_translations in translations.iteritems():
                for case, synonyms in language_translations.iteritems():
                    language_translations[case] = sorted(synonym for synonym in synonyms if synonym)

    def _add_companies_titles(self, groupped_synonyms, all_aviacompanies):
        for aviacompany in all_aviacompanies:
            company_synonyms = groupped_synonyms.setdefault(aviacompany.pk, {})
            for language, _ in settings.LANGUAGES:
                company_language_synonyms = company_synonyms.setdefault(language, {})
                company_language_synonyms.setdefault('nom', set()).add(aviacompany.get_title(language))

    def _group_synonyms(self, all_synonyms):
        groupped_synonyms = dict()

        for synonym in all_synonyms:
            company_synonyms = groupped_synonyms.setdefault(synonym['company_id'], {})
            company_language_synonyms = company_synonyms.setdefault(synonym['language'], {})
            company_language_synonyms.setdefault('nom', set()).add(synonym['synonym'])

        return groupped_synonyms


airlines_synonyms_view = AirlinesSynonymsView(
    airline_repository=airline_repository,
    logger=getLogger(__name__)
)
