# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import
from logging import getLogger

from marshmallow import Schema, fields

from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.backend.repository import flights_to_city_airlines_repository, airline_repository


class FlightsToCityAirlinesSchema(Schema):
    lang = fields.String(required=True)
    national_version = fields.String(required=True)
    settlement_id = fields.Integer(required=True)


class FlightsToCityAirlinesView(CommonView):
    AIRLINES_COUNT = 6

    def __init__(self, airline_repository, flights_to_city_repository, logger):
        self._airline_repository = airline_repository
        self._flights_to_city_repository = flights_to_city_repository
        super(FlightsToCityAirlinesView, self).__init__(FlightsToCityAirlinesSchema(), logger)

    def _process(self, parsed_data):
        settlement_id = parsed_data['settlement_id']
        national_version = parsed_data['national_version']

        airlines = self._get_flights_to_city_airlines(settlement_id, national_version)

        return [{
            u'id': airline.pk,
            u'slug': airline.slug,
            u'score': score,
            u'title':  airline.get_title_with_code(parsed_data['lang']),
            u'logo': airline.logo,

        } for airline, score in airlines]

    def _get_flights_to_city_airlines(self, settlement_id, nv):
        relevant_city_airlines = self._flights_to_city_repository.get_top_n(
            settlement_id, nv, self.AIRLINES_COUNT
        )
        airlines = []
        for al in relevant_city_airlines:
            airline = self._airline_repository.get(al['airline_id'])
            if not airline:
                self._logger.warning('No airline for id %d', al['airline_id'])
                continue
            airlines.append((airline, al['score']))
        return airlines


flights_to_city_airlines_view = FlightsToCityAirlinesView(
    airline_repository, flights_to_city_airlines_repository, getLogger(__name__)
)
