# -*- coding: utf-8 -*-
from __future__ import absolute_import

from logging import getLogger

from marshmallow import Schema, fields

from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.backend.repository import similar_airlines_repository, airline_repository


class SimilarAirlinesSchema(Schema):
    national_version = fields.String(required=True)
    lang = fields.String(required=True)
    company_id = fields.Integer(required=True)


class SimilarAirlinesView(CommonView):
    AIRLINES_COUNT = 6

    def __init__(self, airline_repository, similar_airlines_repository, logger):
        self._airline_repository = airline_repository
        self._similar_airlines_repository = similar_airlines_repository
        super(SimilarAirlinesView, self).__init__(SimilarAirlinesSchema(), logger)

    def _process(self, parsed_data):
        company_id = parsed_data[u'company_id']
        if not self._airline_repository.get(company_id):
            return []

        airlines = self._get_similar_airlines(company_id)
        airline_ids = {al.pk for al, score in airlines}

        if len(airlines) < self.AIRLINES_COUNT:
            for al in self._airline_repository.get_top(parsed_data[u'national_version']):
                if al.pk not in airline_ids and al.pk != company_id:
                    airlines.append((al, 0))
                if len(airlines) >= self.AIRLINES_COUNT:
                    break

        return [{
            u'id': airline.pk,
            u'slug': airline.slug,
            u'score': score,
            u'title':  airline.get_title_with_code(parsed_data['lang']),
            u'logo': airline.logo,

        } for airline, score in airlines]

    def _get_similar_airlines(self, company_id):
        similar_airlines = self._similar_airlines_repository.get(company_id)
        airlines = []
        for al in similar_airlines:
            airline = self._airline_repository.get(al['company_id'])
            if not airline:
                self._logger.warning('No airline for id %d', al['company_id'])
                continue
            airlines.append((airline, al['score']))
        return airlines


similar_airlines_view = SimilarAirlinesView(
    airline_repository, similar_airlines_repository, getLogger(__name__)
)
