# -*- coding: utf-8 -*-
from __future__ import absolute_import

import abc
from logging import Logger  # noqa

from marshmallow import Schema, fields

from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.backend.repository.station import StationRepository, StationModel  # noqa
from travel.avia.backend.repository.settlement import SettlementRepository  # noqa
from travel.avia.backend.repository.country import CountryRepository  # noqa
from travel.avia.backend.repository.station_type import StationTypeRepository  # noqa


class AirportForm(Schema):
    lang = fields.String(missing='ru')


class AirportAbstractView(CommonView):
    __metaclass__ = abc.ABCMeta

    def __init__(self, form, repository, settlement_repository, country_repository, station_type_repository, logger):
        # type: (AirportForm, StationRepository, SettlementRepository, CountryRepository, StationTypeRepository, Logger) -> None
        super(AirportAbstractView, self).__init__(form, logger)
        self._repository = repository
        self._settlement_repository = settlement_repository
        self._country_repository = country_repository
        self._station_type_repository = station_type_repository

    def _prepare(self, airport, lang):
        # type: (StationModel, str) -> dict
        return {
            'id': airport.pk,
            'code': airport.iata or airport.sirena,
            'iataCode': airport.iata,
            'sirenaCode': airport.sirena,
            'urlTitle': airport.get_url_title(lang),
            'popularTitle': airport.get_popular_title(lang),
            'title': airport.get_title(lang),
            'phraseFrom': airport.get_phrase_from(lang),
            'phraseTo': airport.get_phrase_to(lang),
            'phraseIn': airport.get_title(lang),
            'key': airport.point_key,
            'longitude': airport.longitude,
            'latitude': airport.latitude,
            'timeZoneUtcOffset': airport.time_zone_utc_offset,
            'timeZone': airport.time_zone,
            'stationType': self._prepare_station_type(airport.station_type_id, lang),
            'settlement': self._prepare_settlement(airport.settlement_id, lang),
        } if airport else None

    def _prepare_station_type(self, type_id, lang):
        # type: (int, str) -> dict
        station_type = self._station_type_repository.get(type_id)
        return {
            'title': station_type.get_title(lang),
            'prefix': station_type.get_prefix(lang),
        } if station_type else {}

    def _prepare_settlement(self, settlement_id, lang):
        # type: (int, str) -> dict
        settlement = self._settlement_repository.get(settlement_id)
        return {
            'id': settlement.id,
            'geoId': settlement.geo_id,
            'title': settlement.get_title(lang),
            'longitude': settlement.longitude,
            'latitude': settlement.latitude,
            'country': self._prepare_country(settlement.country_id, lang),
        } if settlement else {}

    def _prepare_country(self, country_id, lang):
        # type: (int, str) -> dict
        country = self._country_repository.get(country_id)
        return {
            'geoId': country.geo_id,
            'title': country.get_title(lang),
        } if country else {}
