# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import
from logging import getLogger

from marshmallow import Schema, fields, ValidationError

from travel.avia.library.python.common.models.geo import Country, Region, Station, Settlement
from travel.avia.backend.main.api.fields import PointKey
from travel.avia.backend.main.rest.helpers import CommonView, NotFoundError
from travel.avia.backend.repository import (
    country_repository,
    covid_info_repository,
    region_repository,
    settlement_repository,
    station_repository,
)


class CovidInfoSchema(Schema):
    point_key = fields.String(required=True)
    debug = fields.Boolean(required=False, missing=False)


class CovidInfoView(CommonView):

    def __init__(self, country_repository, region_repository, settlement_repository, station_repository, covid_info_repository, logger):
        super(CovidInfoView, self).__init__(CovidInfoSchema(), logger)
        self.country_repository = country_repository
        self.region_repository = region_repository
        self.settlement_repository = settlement_repository
        self.station_repository = station_repository
        self.covid_info_repository = covid_info_repository

    def _process(self, parsed_data):
        point_key = parsed_data['point_key']
        is_debug = parsed_data.get('debug')

        point_type = None
        point_id = None
        try:
            point_type, point_id = PointKey().deserialize(point_key)
            point_id = int(point_id)
        except ValueError:
            raise ValidationError('Wrong point_key format')

        country_id = None
        settlement_id = None
        settlement_title_ru = None
        if point_type is Settlement:
            settlement = self.settlement_repository.get(point_id)
            if not settlement:
                raise NotFoundError(u'Can not find settlement with id: %s' % point_id)
            if not settlement.country_id:
                raise NotFoundError(u'Can not find country for the settlement with id: %s' % point_id)
            settlement_id = settlement.id
            settlement_title_ru = settlement.get_title('ru')
            country_id = settlement.country_id
        elif point_type is Station:
            station = self.station_repository.get(point_id)
            if not station:
                raise NotFoundError(u'Can not find station with id: %s' % point_id)
            if not station.country_id:
                raise NotFoundError(u'Can not find country for the station with id: %s' % point_id)
            country_id = station.country_id
        elif point_type is Region:
            region = self.region_repository.get(point_id)
            if not region:
                raise NotFoundError(u'Can not find region with id: %s' % point_id)
            if not region.country_id:
                raise NotFoundError(u'Can not find country for the station with id: %s' % point_id)
            country_id = region.country_id
        elif point_type is Country:
            country_id = point_id

        if not country_id:
            raise NotFoundError(u'Can not determine a country for the point with id: %s' % point_id)

        country = self.country_repository.get(country_id)
        if not country:
            raise NotFoundError(u'Can not find country with id %s for point_key %s' % (country_id, point_key))

        covid_info = self.covid_info_repository.get(country_id)
        if not covid_info:
            raise NotFoundError(u'Can not find covid info for a country with id: %s' % country_id)

        result = {
            'country_id': covid_info.country_id,
            'country_title_ru': country.get_title('ru'),
            'tourism': self.get_ternary_value(covid_info.tourism),
            'quarantine': self.get_ternary_value(covid_info.quarantine),
            'quarantine_days': covid_info.quarantine_days or 0,
            'visa': self.get_ternary_value(covid_info.visa),
            'avia': self.get_ternary_value(covid_info.avia),
        }
        if settlement_id == 10622:
            # RASPTICKETS-21501: dirty hack for Phuket
            result['quarantine'] = self.get_ternary_value(False)
            result['quarantine_days'] = 0
            if settlement_title_ru:
                result['country_title_ru'] = settlement_title_ru
        if is_debug:
            if covid_info.updated_at:
                result['updated_at'] = covid_info.updated_at.isoformat()
            if covid_info.comment:
                result['comment'] = covid_info.comment

        return result

    def get_ternary_value(self, nullable_bool_value):
        # type: (bool) -> str
        if nullable_bool_value is True:
            return 'yes'
        if nullable_bool_value is False:
            return 'no'
        return 'unknown'


covid_info_view = CovidInfoView(
    country_repository=country_repository,
    region_repository=region_repository,
    settlement_repository=settlement_repository,
    station_repository=station_repository,
    covid_info_repository=covid_info_repository,
    logger=getLogger(__name__),
)
