# -*- coding: utf-8 -*-
from __future__ import absolute_import

from abc import abstractproperty, ABCMeta
from logging import getLogger, Logger  # noqa
from marshmallow import Schema, fields

from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.backend.repository.country import CountryRepository, country_repository  # noqa


class CountriesSchema(Schema):
    lang = fields.String(required=False, missing='ru')
    fields = fields.Function(
        deserialize=lambda v: set(filter(None, v.split(','))), missing=''
    )


class CountrySchema(CountriesSchema):
    country_id = fields.Integer(required=True)


class AbstractCountryModelView(CommonView):
    __metaclass__ = ABCMeta
    SCHEME = abstractproperty()

    def __init__(self, country_repository, logger):
        # type: (CountryRepository, Logger) -> None
        self._country_repository = country_repository
        super(AbstractCountryModelView, self).__init__(self.SCHEME, logger)

    def data(self, country_id, fields, lang):
        country = self._country_repository.get(country_id)
        data = {
            'id': country.pk,
            'point_key': country.point_key,
            'code': country.code,
            'title': country.get_title(lang),
            'geo_id': country.geo_id,
        }

        return {k: data[k] for k in fields} if fields else data


class CountriesView(AbstractCountryModelView):
    SCHEME = CountriesSchema()

    def _process(self, parsed_data):
        # type: (dict) -> dict
        airlines = self._country_repository.get_all()
        return {
            a.pk: self.data(
                a.pk, parsed_data['fields'], parsed_data['lang']
            ) for a in airlines
        }


class CountryView(AbstractCountryModelView):
    SCHEME = CountrySchema()

    def _process(self, parsed_data):
        # type: (dict) -> dict
        return self.data(
            parsed_data['country_id'], parsed_data['fields'],
            parsed_data['lang']
        )


countries_view = CountriesView(country_repository, getLogger(__name__))

country_view = CountryView(country_repository, getLogger(__name__))
