# -*- coding: utf-8 -*-
from __future__ import absolute_import
from logging import getLogger

from django.conf import settings
from marshmallow import Schema, fields, validate, ValidationError, validates_schema

from travel.avia.library.python.common.models.geo import Station, Settlement
from travel.avia.backend.main.api.fields import PointKey
from travel.avia.backend.main.rest.helpers import CommonView, NotFoundError
from travel.avia.backend.repository import (
    country_repository,
    region_repository,
    settlement_repository,
    station_repository,
)

LANG_ALLOWED = [lang[0] for lang in settings.LANGUAGES]


def _validate_point_key(point_key):
    pk = PointKey()
    try:
        type_pk, point_id = pk.deserialize(point_key)
    except ValueError:
        raise ValidationError('Wrong point_key format')
    if type_pk is not Settlement and type_pk is not Station:
        raise ValidationError('Wrong point_key type, expected s... or c...')
    try:
        int(point_id)
    except ValueError:
        raise ValidationError(u'Wrong point_key format, id expected to be an integer, id: %s' % point_id)


class PointSchema(Schema):
    key = fields.String(required=False, validate=_validate_point_key)
    lang = fields.String(required=False, missing='ru', validate=validate.OneOf(LANG_ALLOWED))
    geo_id = fields.Int(required=False)

    @validates_schema
    def exactly_one_id_is_given(self, data):
        ids = ['key', 'geo_id']
        if not self.exactly_one(data.get(i) for i in ids):
            raise ValidationError('Exactly one of the fields: [{}] is required'.format(', '.join(ids)), ids)

    @staticmethod
    def exactly_one(iterable):
        i = iter(iterable)
        return any(i) and not any(i)

    def __init__(self, *args, **kwargs):
        super(PointSchema, self).__init__(*args, **kwargs)


class PointView(CommonView):
    def __init__(
        self,
        form,
        country_repository,
        region_repository,
        settlement_repository,
        station_repository,
        logger,
    ):
        super(PointView, self).__init__(form, logger)
        self.country_repository = country_repository
        self.region_repository = region_repository
        self.settlement_repository = settlement_repository
        self.station_repository = station_repository

    def _get_settlement_by_point_key(self, point_key, lang):
        field = PointKey()
        point_type, point_id = field.deserialize(point_key)
        point_id = int(point_id)
        settlement = None
        station = None
        if point_type is Settlement:
            settlement = self.settlement_repository.get(point_id)
            if not settlement:
                raise NotFoundError(u'Can not find settlement with id: %s' % point_id)
        elif point_type is Station:
            station = self.station_repository.get(point_id)
            if not station:
                raise NotFoundError(u'Can not find station with id: %s' % point_id)
            settlement = self.settlement_repository.get(station.settlement_id)

        return self._prepare_result(lang, settlement, station)

    def _get_settlement_by_geo_id(self, geo_id, lang):
        settlement = self.settlement_repository.get_by_geo_id(geo_id)
        if not settlement:
            raise NotFoundError(u'Can not find settlement with geo_id: %s' % geo_id)

        return self._prepare_result(lang, settlement)

    def _process(self, parsed_data):
        if 'geo_id' in parsed_data:
            return self._get_settlement_by_geo_id(parsed_data['geo_id'], parsed_data['lang'])
        return self._get_settlement_by_point_key(parsed_data['key'], parsed_data['lang'])

    def _prepare_result(self, lang, settlement=None, station=None):
        region_id = (settlement.region_id if settlement else None) or (station.region_id if station else None)
        country_id = (settlement.country_id if settlement else None) or (station.country_id if station else None)
        region = self.region_repository.get(region_id) if region_id else None
        country = self.country_repository.get(country_id) if country_id else None
        return {
            'settlement': settlement.prepare(lang) if settlement else None,
            'station': self._prepare_station(station, lang) if station else None,
            'region': region.prepare(lang) if region else None,
            'country': country.prepare(lang) if country else None,
        }

    @staticmethod
    def _prepare_station(station, lang):
        if not station:
            return None

        return {
            'title': station.get_title(lang),
            'popularTitle': station.get_popular_title(lang),
            'phraseFrom': station.get_phrase_from(lang),
            'urlTitle': station.get_url_title(lang),
            'phraseIn': station.get_phrase_in(lang),
            'iataCode': station.iata,
            'sirenaCode': station.sirena,
            'phraseTo': station.get_phrase_to(lang),
            'id': station.pk,
            'transportType': station.transport_type,
            'regionId': station.region_id,
            'countryId': station.country_id,
            'longitude': station.longitude,
            'latitude': station.latitude,
        }


point_view = PointView(
    form=PointSchema(),
    country_repository=country_repository,
    region_repository=region_repository,
    settlement_repository=settlement_repository,
    station_repository=station_repository,
    logger=getLogger(__name__),
)
