from __future__ import absolute_import

import abc
import ujson
from flask import Response, request
from marshmallow import ValidationError, fields


class CommonView(object):
    __metaclass__ = abc.ABCMeta

    def __init__(self, form, logger):
        self._form = form
        self._logger = logger

    def _return_json(self, data, status):
        return Response(
            response=ujson.dumps(
                data,
                ensure_ascii=False
            ),
            status=status,
            mimetype="application/json"
        )

    def process(self, *args, **kwargs):
        try:
            query_args = dict(request.args.items())
            intersection = set(kwargs) & set(query_args)
            if intersection:
                self._logger.warning('Duplicated params: %s', intersection)

            query_args.update(kwargs)

            return self._unsafe_process(query_args)
        except NotFoundError as e:
            return self._return_json({
                'data': str(e),
                'status': 'error',
            }, status=404)
        except ValidationError as e:
            return self._return_json({
                'data': str(e),
                'status': 'error',
            }, status=400)
        except Exception as e:
            self._logger.exception(e)
            message = 'can not save message'
            try:
                message = str(e)
            except Exception:
                self._logger.exception(message)
                pass

            return self._return_json({
                'data': message,
                'status': 'error'
            }, status=500)

    def _unsafe_process(self, params):
        try:
            data, errors = self._form.load(params)
        except ValidationError as exception:
            data, errors = None, exception.message

        if errors:
            return self._return_json({
                'data': errors,
                'status': 'error'
            }, status=400)

        return self._return_json({
            'data': self._process(data),
            'status': 'ok'
        }, status=200)

    @abc.abstractmethod
    def _process(self, parsed_form):
        raise NotImplementedError()


class IdsField(fields.Field):
    def _deserialize(self, value, attr, data):
        result = []
        for s in value.split(','):
            try:
                result.append(int(s))
            except ValueError:
                pass

        return result


class NotFoundError(Exception):
    pass
