# -*- coding: utf-8 -*-
from __future__ import absolute_import

from logging import getLogger, Logger  # noqa

from marshmallow import Schema, fields, validates_schema, ValidationError, validate

from travel.avia.backend.main import settings
from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.backend.repository.settlement import SettlementRepository, settlement_repository, SettlementGeoIndex, settlement_geo_index  # noqa

LANG_ALLOWED = [lang[0] for lang in settings.LANGUAGES]


class NearbyAviaSettlementsForm(Schema):
    settlement_id = fields.Integer(
        required=False,
        validate=validate.Range(min=1, error='settlement_id must be positive integer'),
    )
    geo_id = fields.Integer(
        required=False,
        validate=validate.Range(min=1, error='geo_id must be positive integer'),
    )
    radius = fields.Integer(
        required=False,
        missing=200,
        validate=validate.Range(min=1, max=600, error='radius must be in range [1, 600]'),
    )
    lang = fields.String(required=False, missing='ru', validate=validate.OneOf(LANG_ALLOWED))
    include_current_settlement = fields.Boolean(
        required=False,
        missing=False,
    )

    @validates_schema
    def not_empty(self, data):
        if all(data.get(i) is None for i in ('settlement_id', 'geo_id')):
            raise ValidationError('At least once field of [settlement_id, geo_id] must be passed')


class NearbyAviaSettlementsView(CommonView):
    def __init__(self, form, settlement_repository, settlement_geo_index, logger):
        """
        :type form: NearbyAviaSettlementsForm
        :type settlement_repository: SettlementRepository
        :type settlement_geo_index: SettlementGeoIndex
        :type logger: Logger
        """
        super(NearbyAviaSettlementsView, self).__init__(form, logger)
        self._settlement_repository = settlement_repository
        self._settlement_geo_index = settlement_geo_index

    def _process(self, parsed_data):
        original_settlement = None
        settlement_id = parsed_data.get('settlement_id')
        include_current_settlement = parsed_data['include_current_settlement']
        geo_id = parsed_data.get('geo_id')
        radius = parsed_data['radius']
        lang = parsed_data['lang']
        if settlement_id is not None:
            original_settlement = self._settlement_repository.get(settlement_id)
        elif geo_id is not None:
            original_settlement = self._settlement_repository.get_by_geo_id(geo_id)
        if not original_settlement or not all((original_settlement.latitude, original_settlement.longitude)):
            return []
        return [
            self._build_response_item(settlement, distance, lang)
            for settlement, distance in self._settlement_geo_index.get_nearest_with_distance(
                original_settlement.id,
                radius,
                include_current_settlement,
            )
        ]

    def _build_response_item(self, settlement, distance, lang):
        item = settlement.prepare(lang)
        item['distance'] = distance
        return item


default_logger = getLogger(__name__)

nearby_avia_settlements = NearbyAviaSettlementsView(
    form=NearbyAviaSettlementsForm(),
    settlement_repository=settlement_repository,
    settlement_geo_index=settlement_geo_index,
    logger=default_logger,
)
