from __future__ import absolute_import

import re
from logging import Logger, getLogger

from marshmallow import Schema, fields, validate

from travel.avia.library.python.avia_data.models.popular_partners import PopularPartnersByRoute

from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.backend.main.rest.partners.helpers import (
    PartnerPopularityProvider,
    route_partner_popularity_provider
)
from travel.avia.backend.repository.partner import PartnerRepository, partner_repository

POINT_KEY_RE = re.compile(r'^[{}]\d+$'.format(''.join(PopularPartnersByRoute.POINT_PREFIX_MAP)))


class PartnerPopularForm(Schema):
    national_version = fields.String(required=True)
    lang = fields.String(required=True)
    from_point_key = fields.String(required=True, validate=validate.Regexp(POINT_KEY_RE, error='should start with s/c'))
    to_point_key = fields.String(required=True, validate=validate.Regexp(POINT_KEY_RE, error='should start with s/c'))


class RoutePartnerPopularityView(CommonView):
    def __init__(self, partner_popularity_provider, partner_repository, logger):
        # type: (PartnerPopularityProvider, PartnerRepository, Logger) -> None
        super(RoutePartnerPopularityView, self).__init__(
            PartnerPopularForm(), logger
        )
        self._partner_popular_provider = partner_popularity_provider
        self._partner_repository = partner_repository

    def _process(self, parsed_data):
        models = self._partner_popular_provider.get_for(
            from_point_key=parsed_data['from_point_key'],
            to_point_key=parsed_data['to_point_key'],
            national_version=parsed_data['national_version']
        )

        return [
            {
                u'partner_code': self._partner_repository.get_by_id(
                    model.partner_id
                ).code,
                u'score': model.score,
            } for model in models if self._partner_repository.get_by_id(
                model.partner_id
            )
        ]


popular_partners_by_route_view = RoutePartnerPopularityView(
    partner_popularity_provider=route_partner_popularity_provider,
    partner_repository=partner_repository,
    logger=getLogger(__name__)
)
