# -*- coding: utf-8 -*-
from __future__ import absolute_import

from logging import getLogger, Logger  # noqa
from datetime import date

from marshmallow import Schema, fields, validate

from travel.avia.backend.main import settings
from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.backend.main.services.price_index import PriceIndex, FromTo
from travel.avia.backend.main.lib.recipes.popular import PopularDirectionFinder
from travel.avia.backend.repository.settlement import SettlementRepository, settlement_repository  # noqa


class PopularDirectionsFromSettlementForm(Schema):
    from_settlement_id = fields.Integer(
        required=True,
        validate=validate.Range(min=1, error='from_settlement_id must be positive integer'),
    )
    to_settlement_id = fields.Integer(
        required=True,
        validate=validate.Range(min=1, error='to_settlement_id must be positive integer'),
    )
    national_version = fields.String(required=True, validate=validate.OneOf(settings.AVIA_NATIONAL_VERSIONS))
    limit = fields.Integer(
        required=False,
        missing=5,
        validate=validate.Range(min=0, error='limit must be non-negative integer'),
    )
    window_size = fields.Integer(
        required=False,
        missing=30,
        validate=validate.Range(min=1, error='window_size must be positive integer'),
    )


class PopularDirectionsFromSettlementView(CommonView):
    def __init__(self, form, settlement_repository, price_index, logger):
        """
        :type form: PopularDirectionsFromSettlementForm
        :type settlement_repository: SettlementRepository
        :type price_index: PriceIndex
        :type logger: Logger
        """
        super(PopularDirectionsFromSettlementView, self).__init__(form, logger)
        self._settlement_repository = settlement_repository
        self._price_index = price_index

    def _process(self, parsed_data):
        from_settlement_id = parsed_data['from_settlement_id']
        to_settlement_id = parsed_data['to_settlement_id']
        national_version = parsed_data['national_version']
        limit = parsed_data['limit']
        window_size = parsed_data['window_size']

        from_settlement = self._settlement_repository.get(from_settlement_id)
        if not from_settlement:
            return []

        # Берем с небольшим запасом на случай, если вдруг в прайс-индексе не будет цен по каким-то направлениям
        popular_arrival_settlement_ids = PopularDirectionFinder.get_top_direction_ids(
            from_settlement,
            national_version,
            limit + 5,
        )
        popular_arrival_settlement_ids = [
            settlement_id
            for settlement_id in popular_arrival_settlement_ids
            if settlement_id != to_settlement_id
        ]
        destination_popularity_index = {
            settlement_id: i for i, settlement_id in enumerate(popular_arrival_settlement_ids)
        }
        directions = [FromTo(from_settlement_id, settlement_id) for settlement_id in popular_arrival_settlement_ids]
        minprices = self._price_index.search(directions, national_version, date.today(), None, (window_size + 1) // 2)
        minprices = sorted(minprices, key=lambda mp: destination_popularity_index[mp['to_id']])
        if len(minprices) < limit:
            self._logger.warning(
                'limit=%d, national version=%s, requested directions=%s, received directions=%s',
                limit,
                national_version,
                directions,
                minprices,
            )
        return minprices[:limit]


default_logger = getLogger(__name__)

popular_directions_from_settlement = PopularDirectionsFromSettlementView(
    form=PopularDirectionsFromSettlementForm(),
    settlement_repository=settlement_repository,
    price_index=PriceIndex(avia_price_index_url=settings.AVIA_PRICE_INDEX_URL, logger=default_logger),
    logger=default_logger,
)
