# -*- coding: utf-8 -*-
from __future__ import absolute_import

from logging import getLogger, Logger  # noqa

from typing import Optional, List  # noqa
from marshmallow import Schema, fields, ValidationError

from travel.avia.library.python.common.utils import environment

from travel.avia.backend.main import settings
from travel.avia.backend.main.rest.helpers import CommonView
from travel.avia.backend.main.services.price_index import PriceIndex, FromTo
from travel.avia.backend import repository
from travel.avia.backend.repository.country import CountryRepository, CountryModel  # noqa
from travel.avia.backend.repository.direction import DirectionRepository, Direction  # noqa
from travel.avia.backend.repository.region import RegionModel, RegionRepository  # noqa
from travel.avia.backend.repository.settlement import SettlementRepository, SettlementModel  # noqa
from travel.avia.backend.repository.settlement_big_image import SettlementBigImageRepository  # noqa


class TravelRecipesForm(Schema):
    from_geo_id = fields.Integer(required=True)
    forward_date = fields.Date(required=False)
    backward_date = fields.Date(required=False)
    national_version = fields.String(required=True)
    limit = fields.Integer(required=False, missing=5)
    lang = fields.String(required=False, missing=u'ru')


class TravelRecipesView(CommonView):
    def __init__(
            self, form, country_repository, region_repository, settlement_repository, direction_repository,
            settlement_big_image_repository, price_index, logger
    ):
        """
        :type form: TravelRecipesForm
        :type country_repository: CountryRepository
        :type region_repository: RegionRepository
        :type settlement_repository: SettlementRepository
        :type direction_repository: DirectionRepository
        :type settlement_big_image_repository: SettlementBigImageRepository
        :type price_index: PriceIndex
        :type logger: Logger
        """
        super(TravelRecipesView, self).__init__(form, logger)
        self._country_repository = country_repository
        self._region_repository = region_repository
        self._settlement_repository = settlement_repository
        self._direction_repository = direction_repository
        self._settlement_big_image_repository = settlement_big_image_repository
        self._price_index = price_index
        self._logger = logger

    def _process(self, parsed_data):
        national_version = parsed_data.get('national_version')
        from_geo_id = parsed_data.get('from_geo_id')
        lang = parsed_data.get('lang')

        from_settlement = self._get_settlement_by_geo_id(from_geo_id)

        (from_region, from_country) = self._get_region_and_country(from_settlement)

        from_ids = []  # type: List[int]
        do_not_fly_there = set()  # исходный город и столица региона, чтобы не показывать рейсы в них из столицы
        if from_settlement:
            from_ids.append(from_settlement.id)
            do_not_fly_there.add(from_settlement.id)

        if from_region:
            region_capital = self._settlement_repository.get_region_capital(from_region.pk)
            if region_capital and region_capital.id not in from_ids:
                from_ids.append(region_capital.id)
                do_not_fly_there.add(region_capital.id)

        if from_country:
            country_capital = self._settlement_repository.get_country_capital(from_country.pk)
            if country_capital and country_capital.id not in from_ids:
                from_ids.append(country_capital.id)

        if national_version:
            # костыль для домена com.tr, per https://st.yandex-team.ru/RASPTICKETS-14128#5d243e2334894f001dd46b78
            national_version_country_code = 'TR' if national_version == 'com.tr' else national_version.upper()
            national_version_country = self._country_repository.get_by_code(national_version_country_code)
            if national_version_country:
                if not from_country or from_country.pk != national_version_country.pk:
                    national_capital = self._settlement_repository.get_country_capital(national_version_country.pk)
                    if national_capital and national_capital.id not in from_ids:
                        from_ids.append(national_capital.id)

        # построим список популярных направлений из выбранных городов / столиц
        directions = []  # type: List[Direction]

        if not from_ids:
            directions += self._direction_repository.get_all(national_version)

        if national_version:
            for city_id in from_ids:
                directions += self._direction_repository.get_from_settlement(national_version, city_id)

        if not directions:
            return []

        # для каждого destination оставим только один direction, и уберём направления в do_not_fly_there
        directions = self._filter_by_destination(directions, do_not_fly_there)

        # получим карту минцен по directions
        forward_date = parsed_data.get('forward_date')
        minprices = self._price_index.search(
            directions=[
                FromTo(from_to.departure_settlement_id, from_to.arrival_settlement_id)
                for from_to in directions
            ],
            national_version=national_version,
            forward_date=forward_date if forward_date else environment.now(),
            backward_date=parsed_data.get('backward_date'),
            window_size=1 if forward_date else 30,
            results_per_direction=1
        )

        # отсортируем цены по такому же порядку городов вылета, как в directions
        from_locations = [request_direction.departure_settlement_id for request_direction in directions]
        minprices = filter(lambda item: item.get('from_id') and item.get('from_id') in from_locations, minprices)
        minprices.sort(key=lambda item: from_locations.index(item.get('from_id')))

        result = [
            self._prepare(minprice, lang) for minprice in minprices[:parsed_data.get('limit')]
        ]
        return filter(None, result)

    def _get_settlement_by_geo_id(self, geo_id):
        found_settlement = self._settlement_repository.get_by_geo_id(geo_id)

        if not found_settlement:
            raise ValidationError('Unknown settlement with geo_id: {}'.format(geo_id))

        return found_settlement

    def _get_region_and_country(self, settlement):
        # type: (SettlementModel) -> (Optional[RegionModel], Optional[CountryModel])
        return self._region_repository.get(settlement.region_id), self._country_repository.get(settlement.country_id)

    @staticmethod
    def _filter_by_destination(directions, do_not_fly_there):
        # type: (List[Direction], set) -> List[Direction]
        result = []  # type: List[Direction]
        to_ids = set(do_not_fly_there)

        for direction in directions:
            if direction.arrival_settlement_id not in to_ids:
                to_ids.add(direction.arrival_settlement_id)
                result.append(direction)

        return result

    def _prepare(self, minprice, lang):
        # type: (dict, str) -> Optional[dict]

        from_id = minprice.get('from_id')
        to_id = minprice.get('to_id')

        if not from_id or not to_id:
            return None

        settlement_from = self._settlement_repository.get(from_id)
        minprice['settlement_from'] = settlement_from.prepare(lang) if settlement_from else None

        settlement_to = self._settlement_repository.get(to_id)
        minprice['settlement_to'] = settlement_to.prepare(lang) if settlement_to else None

        minprice['image'] = self._settlement_big_image_repository.get_image_or_default_url(to_id)

        return minprice


default_logger = getLogger(__name__)

travel_recipes_view = TravelRecipesView(
    form=TravelRecipesForm(),
    country_repository=repository.country.country_repository,
    region_repository=repository.region.region_repository,
    settlement_repository=repository.settlement.settlement_repository,
    direction_repository=repository.direction.direction_repository,
    settlement_big_image_repository=repository.settlement_big_image.settlement_big_image_repository,
    price_index=PriceIndex(avia_price_index_url=settings.AVIA_PRICE_INDEX_URL, logger=default_logger),
    logger=default_logger,
)
