# -*- coding: utf-8 -*-
from __future__ import absolute_import

from collections import namedtuple
from logging import getLogger, Logger  # noqa
from urlparse import urljoin

from typing import List  # noqa
from requests import Session
from requests.adapters import HTTPAdapter
from traceback import format_exc
from urllib3 import Retry

FromTo = namedtuple('FromTo', ['from_id', 'to_id'])


class PriceIndex(object):
    def __init__(self, avia_price_index_url, timeout=5, logger=None):
        # type: (str, int, Logger) -> None
        self._avia_price_index_url = avia_price_index_url

        self._session = Session()
        self._session.mount('http://', HTTPAdapter(max_retries=Retry(
            total=3,
            read=3,
            connect=3,
            backoff_factor=0.1,
            status_forcelist=(500, 502, 503, 504),
        )))
        self._timeout = timeout
        self._logger = logger or getLogger(__name__)

    def search(self, directions, national_version, forward_date, backward_date, window_size=0, results_per_direction=1):
        """
        :type directions: List[FromTo]
        :type national_version: unicode
        :type forward_date: date
        :type backward_date: date
        :type window_size: int
        :type results_per_direction: int
        :returns List[dict]
        """
        if not directions:
            return []

        try:
            request = {
                'directions': [from_to._asdict() for from_to in directions],
                'forward_date': forward_date.isoformat() if forward_date else None,
                'backward_date': backward_date.isoformat() if backward_date else None,
                'window_size': window_size,
                'results_per_direction': results_per_direction
            }
            raw_response = self._session.post(
                urljoin(
                    self._avia_price_index_url,
                    'search_methods/v1/top_directions_by_date_window/{}'.format(national_version)),
                json=request,
                timeout=self._timeout
            )

            raw_response.raise_for_status()
            response = raw_response.json()
            if response['status'] != 'ok':
                self._logger.warning('no min prices, response = %r, directions = %r', response, directions)
                return []

            return response['data']
        except:
            self._logger.warning('exception while calling price-index: %r', format_exc())
            return []
