# -*- coding: utf-8 -*-
from __future__ import absolute_import

from cachetools import cachedmethod
from django.conf import settings
from typing import List, Optional, Dict  # noqa

from travel.avia.library.python.avia_data.models import AviaCompany
from travel.avia.library.python.avia_data.models import CompanyTariff
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.backend.repository.helpers import NationalBox
from travel.avia.backend.repository.translations import TranslatedTitleRepository, translated_title_repository  # noqa

LANGUAGES = {settings.DOMAIN_LANGUAGE_MAP[tld][0] for tld in settings.AVIA_NATIONAL_VERSIONS}


class AirlineModel(object):
    __slots__ = (
        '_translated_title_repository',
        'pk', 'title_id', 'slug', 'tariff',
        'iata', 'sirena', 'icao', 'icao_ru', 'popular_score_by_national_version',
        'seo_description_key', 'logo',
        'baggage_rules', 'baggage_rules_url',
        'registration_url', 'alliance_id', 'logo_bgcolor', 'url',
        'registration_url_locals', 'registration_phone',
        'registration_phone_locals', '_title_id', '_popular_score_by_national_version',
        'baggage_length', 'baggage_width', 'baggage_height', 'baggage_dimensions_sum',
        'carryon_length', 'carryon_width', 'carryon_height', 'carryon_dimensions_sum',
        'hidden'
    )

    def __init__(self, translated_title_repository,
                 pk, title_id, slug, tariff,
                 iata, sirena, icao, icao_ru, popular_score_by_national_version,
                 seo_description_key, logo,
                 baggage_rules, baggage_rules_url,
                 registration_url, alliance_id, logo_bgcolor, url,
                 registration_url_locals, registration_phone,
                 registration_phone_locals,
                 hidden,
                 baggage_length, baggage_width, baggage_height, baggage_dimensions_sum,
                 carryon_length, carryon_width, carryon_height, carryon_dimensions_sum,
                 ):
        """
        :param TranslatedTitleRepository translated_title_repository:
        :param int pk:
        :param int title_id:
        :param str slug:
        :param Optional[Dict] tariff:
        :param Optional[str] iata:
        :param Optional[unicode] sirena:
        :param Optional[str] icao:
        :param Optional[str] icao_ru:
        :param NationalBox popular_score_by_national_version:
        :param Optional[str] seo_description_key:
        :param str logo:
        :param str baggage_rules:
        :param str baggage_rules_url:
        :param str registration_url:
        :param Optional[int] alliance_id:
        :param str logo_bgcolor:
        :param str url:
        :param Optional[Dict] registration_url_locals:
        :param str registration_phone:
        :param Optional[Dict] registration_phone_locals:
        :param bool hidden:
        :param int baggage_length:
        :param int baggage_width:
        :param int baggage_height:
        :param int baggage_dimensions_sum:
        :param int carryon_length:
        :param int carryon_width:
        :param int carryon_height:
        :param int carryon_dimensions_sum:
        """
        self._translated_title_repository = translated_title_repository

        self.pk = pk
        self.slug = slug
        self.tariff = tariff
        self.logo = logo
        self.seo_description_key = seo_description_key
        self.baggage_rules = baggage_rules
        self.baggage_rules_url = baggage_rules_url
        self.registration_url = registration_url
        self.alliance_id = alliance_id
        self.logo_bgcolor = logo_bgcolor
        self.url = url
        self.iata = iata
        self.sirena = sirena
        self.icao = icao
        self.icao_ru = icao_ru
        self.registration_url_locals = registration_url_locals
        self.registration_phone = registration_phone
        self.registration_phone_locals = registration_phone_locals

        self._title_id = title_id
        self._popular_score_by_national_version = popular_score_by_national_version

        self.hidden = hidden

        self.baggage_length = baggage_length
        self.baggage_width = baggage_width
        self.baggage_height = baggage_height
        self.baggage_dimensions_sum = baggage_dimensions_sum

        self.carryon_length = carryon_length
        self.carryon_width = carryon_width
        self.carryon_height = carryon_height
        self.carryon_dimensions_sum = carryon_dimensions_sum

    @property
    def code(self):
        return self.iata or self.sirena

    def get_title(self, lang):
        # type: (str) -> str
        return self._translated_title_repository.get(self._title_id, lang)

    def _get_field_local(self, field_name, lang):
        field_locals = getattr(self, field_name)
        return field_locals.get(lang) if field_locals else None

    def _get_field_local_fallback_to_global(self, field_name, lang):
        field_local = self._get_field_local(field_name + '_locals', lang)
        field_global = getattr(self, field_name)
        return field_local or field_global or None  # don't return '', return None instead

    def get_registration_url(self, lang):
        return self._get_field_local_fallback_to_global('registration_url', lang)

    def get_registration_phone(self, lang):
        return self._get_field_local_fallback_to_global('registration_phone', lang)

    def get_title_with_code(self, lang):
        # type: (str) -> unicode
        title = self.get_title(lang)
        code = self.code
        if not code:
            return title

        return u'{} ({})'.format(
            title,
            code
        )

    def get_popular_score(self, national_version):
        # type: (str) -> int
        return self._popular_score_by_national_version.get(national_version) or 0

    def __repr__(self):
        return str(self.__dict__)

    def __eq__(self, other):
        # type: (AirlineModel) -> bool
        if not (self.pk == other.pk or self.slug == other.slug or
                self._title_id == other._title_id or
                self.sirena == other.sirena or
                self.iata == other.iata):
            return False

        return self._popular_score_by_national_version.__dict__ == other._popular_score_by_national_version.__dict__


class AirlineRepository(object):
    def __init__(self, translated_title_repository):
        # type: (TranslatedTitleRepository) -> None
        self._translated_title_repository = translated_title_repository
        self._index = {}
        self.__top_cache = {}

    def reset(self):
        # type: () -> None
        self._index = {}
        self._translated_title_repository.reset()
        self.__top_cache.clear()

    def pre_cache(self):
        # type: () -> None
        popular_score = 'popular_score_for_{}'.format
        registration_url = 'registration_url_{}'.format
        registration_phone = 'registration_phone_{}'.format
        fields = (
            [
                'id', 'new_L_title_id', 'slug', 'iata', 'sirena_id', 'icao', 'icao_ru', 'svg_logo2',
                'seo_description_i18n', 'seo_description_approved', 'registration_url',
                'registration_phone', 'alliance_id', 'logo_bgcolor', 'url',
                'hidden',
            ] + map(popular_score, settings.AVIA_NATIONAL_VERSIONS)
            + map(registration_url, LANGUAGES)
            + map(registration_phone, LANGUAGES)
        )
        airlines = list(Company.objects.filter(
            t_type_id=TransportType.PLANE_ID,
        ).values(*fields))

        default_tariffs = {
            ct['avia_company_id']: ct
            for ct in CompanyTariff.objects.all().values()
            if self._is_default_tariff(ct)
        }
        avia_data_airlines = {
            b.pop('rasp_company_id'): b
            for b in AviaCompany.objects.all().values(
                'rasp_company_id', 'baggage_rules', 'baggage_rules_are_valid', 'baggage_rules_url', 'iata',
                'baggage_length', 'baggage_width', 'baggage_height', 'baggage_dimensions_sum',
                'carryon_length', 'carryon_width', 'carryon_height', 'carryon_dimensions_sum',
            )
        }
        title_ids = set(c['new_L_title_id'] for c in airlines)
        self._translated_title_repository.fetch(title_ids)

        self._index = {}
        for c in airlines:
            avia_data_airline = avia_data_airlines.get(c['id'], {})
            self._index[c['id']] = AirlineModel(
                translated_title_repository=self._translated_title_repository,
                pk=c['id'],
                title_id=c['new_L_title_id'],
                popular_score_by_national_version=NationalBox({
                    national_version: c['popular_score_for_{}'.format(national_version)]
                    for national_version in settings.AVIA_NATIONAL_VERSIONS
                }),
                slug=c['slug'],
                tariff=default_tariffs.get(c['id']),
                iata=avia_data_airline.get('iata') or c['iata'],
                sirena=c['sirena_id'],
                icao=c['icao'],
                icao_ru=c['icao_ru'],
                logo=c['svg_logo2'],
                seo_description_key=c['seo_description_i18n'] if c['seo_description_approved'] else None,
                baggage_rules=avia_data_airline.get('baggage_rules') if avia_data_airline.get('baggage_rules_are_valid', False) else '',
                baggage_rules_url=avia_data_airline.get('baggage_rules_url'),
                registration_url=c['registration_url'],
                registration_url_locals={lang: c[registration_url(lang)] for lang in LANGUAGES},
                registration_phone=c['registration_phone'],
                registration_phone_locals={lang: c[registration_phone(lang)] for lang in LANGUAGES},
                alliance_id=c['alliance_id'],
                logo_bgcolor=c['logo_bgcolor'],
                url=c['url'],
                hidden=c['hidden'],
                carryon_width=avia_data_airline.get('carryon_width'),
                carryon_length=avia_data_airline.get('carryon_length'),
                carryon_height=avia_data_airline.get('carryon_height'),
                carryon_dimensions_sum=avia_data_airline.get('carryon_dimensions_sum'),
                baggage_width=avia_data_airline.get('baggage_width'),
                baggage_length=avia_data_airline.get('baggage_length'),
                baggage_height=avia_data_airline.get('baggage_height'),
                baggage_dimensions_sum=avia_data_airline.get('baggage_dimensions_sum'),
            )

    def get_all(self):
        # type: () -> List[AirlineModel]
        return self._index.values()

    @cachedmethod(lambda s: s.__top_cache)
    def get_top(self, national_version):
        # type: (str) -> List[AirlineModel]
        top_airlines = self.get_all()
        top_airlines.sort(
            key=lambda a: -a.get_popular_score(national_version),
            reverse=True,
        )
        return top_airlines

    def get(self, company_id):
        # type: (int) -> Optional[AirlineModel]
        return self._index.get(company_id)

    @staticmethod
    def _is_default_tariff(company_tariff):
        return company_tariff['description'] == u'По умолчанию'


airline_repository = AirlineRepository(translated_title_repository)
