from __future__ import absolute_import

from typing import Optional, List  # noqa
from operator import attrgetter

from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.common.models.transport import TransportType
from travel.avia.library.python.common.utils.iterrecipes import group_by


class CompanyModel(object):
    __slots__ = (
        'id',
        'iata',
        'sirena_id',
        'icao',
        'icao_ru',
        't_type_id',
        'priority'
    )

    def __init__(self, pk, iata, sirena_id, icao, icao_ru, t_type_id, priority):
        # type: (int, Optional[str], Optional[str], Optional[str], Optional[str], int, int) -> None
        self.id = pk
        self.iata = iata
        self.sirena_id = sirena_id
        self.icao = icao
        self.icao_ru = icao_ru
        self.t_type_id = t_type_id
        self.priority = priority

    def __repr__(self):
        return u'<CompanyModel id={} iata={} sirena_id={} icao={} icao_ru={} t_type_id={} priority={}>'.format(
            self.id,
            self.iata,
            self.sirena_id,
            self.icao,
            self.icao_ru,
            self.t_type_id,
            self.priority,
        )


class CompanyRepository(object):
    def __init__(self):
        # type: () -> None
        self._index = {}
        self._companies_by_iata = {}
        self._companies_by_sirena = {}
        self._companies_by_icao = {}
        self._companies_by_icao_ru = {}

    def pre_cache(self):
        # type: () -> None
        fields = [
            'id',
            'iata',
            'sirena_id',
            'icao',
            'icao_ru',
            't_type_id',
            'priority',
        ]
        companies = list(Company.objects.filter(
            t_type_id=TransportType.PLANE_ID,
            hidden=False,
        ).values(*fields))

        index = {}

        for c in companies:
            pk = c['id']
            m = CompanyModel(pk, c['iata'], c['sirena_id'], c['icao'], c['icao_ru'], c['t_type_id'], c['priority'])
            index[pk] = m

        self._index = index
        self._companies_by_iata = self._group_companies_by_attribute('iata')
        self._companies_by_sirena = self._group_companies_by_attribute('sirena_id')
        self._companies_by_icao = self._group_companies_by_attribute('icao')
        self._companies_by_icao_ru = self._group_companies_by_attribute('icao_ru')

    def _group_companies_by_attribute(self, attribute):
        # type: (str) -> dict
        return {
            attr_value: sorted(cs, key=attrgetter('priority'), reverse=True)
            for attr_value, cs in group_by(self._index.values(), attrgetter(attribute)) if attr_value
        }

    def get_companies_by_iata(self):
        # type: () -> dict
        return self._companies_by_iata

    def get_companies_by_sirena(self):
        # type: () -> dict
        return self._companies_by_sirena

    def get_companies_by_icao(self):
        # type: () -> dict
        return self._companies_by_icao

    def get_companies_by_icao_ru(self):
        # type: () -> dict
        return self._companies_by_icao_ru

    def get_companies_by_id(self):
        # type: () -> dict
        return self._index

    def get_all(self):
        # type: () -> List[CompanyModel]
        return self._index.values()


company_repository = CompanyRepository()
