from __future__ import absolute_import

from typing import Optional, List  # noqa

from unidecode import unidecode

from travel.avia.library.python.common.models.geo import Country
from travel.avia.backend.repository.translations import TranslatedTitleRepository, translated_title_repository  # noqa


class CountryModel(object):
    __slots__ = (
        '_translated_title_repository',
        'pk', 'point_key', '_title_id', 'geo_id', 'code',
    )

    def __init__(self, translated_title_repository,
                 pk, title_id, geo_id, code):
        # type: (TranslatedTitleRepository, int, int, Optional[int], Optional[str]) -> None
        self._translated_title_repository = translated_title_repository

        self.pk = pk
        self.point_key = u'l{}'.format(pk)

        self._title_id = title_id

        self.geo_id = geo_id
        self.code = code

    def get_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get(self._title_id, lang)

    def get_genitive_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get_genitive(
            self._title_id, lang
        )

    def get_accusative_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get_accusative(
            self._title_id, lang
        )

    def get_url_title(self, lang):
        # type: (str) -> str
        return unidecode(self.get_title(lang=lang)).replace("'", '')

    def prepare(self, lang):
        # type: (str) -> dict
        return {
            'id': self.pk,
            'title': self.get_title(lang),
            'geoId': self.geo_id,
            'code': self.code,
        }

    def __repr__(self):
        return u"<CountryModel pk={} title_ru={} title_en={}>".format(
            self.pk,
            self.get_title('ru'),
            self.get_title('en'),
        )


class CountryRepository(object):
    def __init__(self, translated_title_repository):
        # type: (TranslatedTitleRepository) -> None
        self._translated_title_repository = translated_title_repository

        self._index = {}
        self._geo_id_index = {}
        self._code_index = {}

    def _load_db_models(self):
        # type: () -> List[dict]
        fields = [
            'id',
            'new_L_title_id',
            '_geo_id',
            'code'
        ]

        return list(Country.objects.values(*fields))

    def pre_cache(self):
        # type: () -> None
        counties = self._load_db_models()

        title_ids = set(c['new_L_title_id'] for c in counties)
        self._translated_title_repository.fetch(title_ids)

        index = {}
        geo_id_index = {}
        code_index = {}
        for s in counties:
            pk = s['id']
            geo_id = s['_geo_id']

            m = CountryModel(
                translated_title_repository=self._translated_title_repository,

                pk=pk,
                title_id=s['new_L_title_id'],

                geo_id=geo_id,
                code=s['code'],
            )

            index[pk] = m
            if geo_id is not None:
                geo_id_index[geo_id] = m

            if m.code is not None:
                code_index[m.code] = m

        self._index = index
        self._geo_id_index = geo_id_index
        self._code_index = code_index

    def get(self, country_id):
        # type: (int) -> Optional[CountryModel]
        return self._index.get(country_id)

    def get_by_geo_id(self, geo_id):
        # type: (int) -> Optional[CountryModel]
        return self._geo_id_index.get(geo_id)

    def get_by_code(self, code):
        # type: (str) -> Optional[CountryModel]
        return self._code_index.get(code)

    def get_all(self):
        # type: () -> List[CountryModel]
        return self._index.values()


country_repository = CountryRepository(translated_title_repository)
