from __future__ import absolute_import

from datetime import datetime
from logging import Logger, getLogger
from typing import Optional, List  # noqa

from travel.avia.library.python.common.models.country_covid_info import CountryCovidInfo


class CountryCovidInfoModel(object):
    __slots__ = (
        'pk', 'tourism', 'quarantine', 'quarantine_days', 'visa', 'avia', 'updated_at', 'comment', 'country_id'
    )

    def __init__(self, pk, tourism, quarantine, quarantine_days, visa, avia, updated_at, comment, country_id):
        # type: (int, Optional[bool], Optional[bool], Optional[int], Optional[bool], Optional[bool], Optional[datetime], Optional[str], int) -> None
        self.pk = pk
        self.tourism = tourism
        self.quarantine = quarantine
        self.quarantine_days = quarantine_days
        self.visa = visa
        self.avia = avia
        self.updated_at = updated_at
        self.comment = comment
        self.country_id = country_id

    def __repr__(self):
        return u"<CountryCovidInfoModel pk={} country_id={}>".format(
            self.pk,
            self.country_id,
        )


class CovidInfoRepository(object):
    def __init__(self, logger=None):
        # type: (Logger) -> None
        self._index = {}
        self._logger = logger

    def _load_db_models(self):
        # type: () -> List[dict]
        fields = [
            'id',
            'tourism',
            'quarantine',
            'quarantine_days',
            'visa',
            'avia',
            'updated_at',
            'comment',
            'country_id',
        ]

        return list(CountryCovidInfo.objects.values(*fields))

    def pre_cache(self):
        # type: () -> None
        if self._logger:
            self._logger.info('Pre-caching covid restrictions info')

        covid_infos = self._load_db_models()

        index = {}
        for ci in covid_infos:
            pk = ci['id']
            country_id = ci['country_id']

            model = CountryCovidInfoModel(
                pk=pk,
                tourism=ci['tourism'],
                quarantine=ci['quarantine'],
                quarantine_days=ci['quarantine_days'],
                visa=ci['visa'],
                avia=ci['avia'],
                updated_at=ci['updated_at'],
                comment=ci['comment'],
                country_id=country_id,
            )

            index[country_id] = model
        self._index = index

    def get(self, country_id):
        # type: (int) -> Optional[CountryCovidInfoModel]
        return self._index.get(country_id)


covid_info_repository = CovidInfoRepository(logger=getLogger(__name__))
