from __future__ import absolute_import

from typing import List  # noqa

from travel.avia.library.python.avia_data.models import Currency

from travel.avia.backend.repository.currency_translation import (  # noqa
    CurrencyTranslationRepository, currency_translation_repository, CurrencyTranslationModel
)


class CurrencyModel(object):
    __slots__ = ('_translation_repository', 'pk', 'code', 'iso_code')

    def __init__(self, translation_repository, pk, code, iso_code):
        # type: (CurrencyTranslationRepository, int, unicode, unicode) -> None
        self._translation_repository = translation_repository

        self.pk = pk
        self.code = code
        self.iso_code = iso_code

    def _get_translation(self, lang):
        # type: (str) -> CurrencyTranslationModel
        return self._translation_repository.get_translation(
            currency_id=self.pk,
            lang=lang
        )

    def get_title(self, lang):
        # type: (str) -> str
        return self._get_translation(
            lang=lang
        ).title

    def get_title_in(self, lang):
        # type: (str) -> str
        return self._get_translation(
            lang=lang
        ).title_in

    def get_template(self, lang):
        # type: (str) -> str
        return self._get_translation(
            lang=lang
        ).template

    def get_template_whole(self, lang):
        # type: (str) -> str
        return self._get_translation(
            lang=lang
        ).template_whole

    def get_template_cents(self, lang):
        # type: (str) -> str
        return self._get_translation(
            lang=lang
        ).template_cents

    def __repr__(self):
        return u"<CurrencyModel pk={} code={}>".format(self.pk, self.code)


class CurrencyRepository(object):
    def __init__(self, translation_repository):
        # type: (CurrencyTranslationRepository) -> None
        self._translation_repository = translation_repository

        self._by_id = None
        self._by_code = None
        self._by_iso_code = None

    def fetch(self):
        # type: () -> None
        currencies = [
            CurrencyModel(
                translation_repository=self._translation_repository,
                pk=c['id'],
                code=c['code'],
                iso_code=c['iso_code'],
            ) for c in Currency.objects.values('id', 'code', 'iso_code')
        ]

        by_id = {}
        by_code = {}
        by_iso_code = {}

        for c in currencies:
            by_id[c.pk] = c
            by_code[c.code] = c
            by_iso_code[c.iso_code] = c

        self._by_id = by_id
        self._by_code = by_code
        self._by_iso_code = by_iso_code

    def get_all_codes(self):
        # type: () -> List[str]
        return self._by_code.keys()

    def get_all(self):
        # type: () -> List[CurrencyModel]
        return self._by_code.values()

    def get_iso_codes(self):
        # type: () -> List[str]
        return self._by_code.keys()

    def get_by_id(self, pk):
        # type: (int) -> CurrencyModel
        return self._by_id[pk]

    def get_by_code(self, code):
        # type: (str) -> CurrencyModel
        return self._by_code[code]

    def get_by_iso_code(self, iso_code):
        # type: (str) -> CurrencyModel
        return self._by_iso_code[iso_code]


currency_repository = CurrencyRepository(
    currency_translation_repository
)
