# coding=utf-8
from __future__ import absolute_import

from logging import Logger, getLogger  # noqa
from datetime import timedelta
from typing import Optional, List  # noqa

from django.conf import settings

from travel.avia.library.python.avia_data.libs.currency.rates import dohop_converter, converter
from travel.avia.library.python.common.utils import environment

from travel.avia.backend.repository.currency import CurrencyRepository, currency_repository, CurrencyModel  # noqa


class CurrencyRatesRepository(object):
    CACHE_TIME = timedelta(hours=1)

    def __init__(self, currency_repository, rates_provider,
                 dohop_rates_provider, environment, logger):
        # type: (CurrencyRepository, any, any, any, Logger) -> None
        self._currency_repository = currency_repository
        self._rates_provider = rates_provider
        self._dohop_rates_provider = dohop_rates_provider
        self._environment = environment
        self._logger = logger

        self._index = dict.fromkeys(settings.AVIA_NATIONAL_VERSIONS)
        self._last_update = None

    def _fetch_for(self, nv):
        currency_codes = self._currency_repository.get_all_codes()
        base_currency_code = settings.AVIA_NATIONAL_CURRENCIES[nv]
        geo_id = settings.NATIONAL_CURRENCY_RATES_GEOID[nv]

        src, rates_by_code = self._rates_provider.fetch_rates(
            currencies_codes=currency_codes,
            geoid=geo_id,
            base_currency_code=base_currency_code
        )

        if not src and nv == 'com':
            src, rates_by_code = self._dohop_rates_provider.fetch_rates(
                currency_codes,
                settings.AVIA_NATIONAL_CURRENCIES[nv]
            )

        if not src:
            raise Exception('can not fetch fresh rates for {}'.format(nv))

        fetched_currency_codes = set(rates_by_code.keys())

        currency_codes = set(currency_codes)
        unknown_currencies = fetched_currency_codes - currency_codes
        if unknown_currencies:
            self._logger.warn(
                'Fetched unknown currencies %r', unknown_currencies
            )

        unfetched_currency_codes = currency_codes - fetched_currency_codes
        if unfetched_currency_codes:
            self._logger.warn(
                'UnFetched currencies %r', unfetched_currency_codes
            )

        rates_by_currency = {}
        for code in fetched_currency_codes & currency_codes:
            rates_by_currency[code] = rates_by_code[code]

        self._index[nv] = rates_by_currency

    def fetch(self, national_versions=None):
        # type: (Optional[List[str]]) -> None
        if not national_versions:
            national_versions = settings.AVIA_NATIONAL_VERSIONS

        for nv in national_versions:
            for retry in range(3):
                try:
                    self._fetch_for(nv)

                    break
                except Exception as e:
                    self._logger.exception('Can not fetch fresh rates %r', e)
            else:
                if self._index[nv]:
                    self._logger.warn('Use previous data')
                else:
                    self._logger.critical(
                        'Can not fetch rates for %r after 3 retries',
                        nv
                    )
                    self._index = {
                        'ru': {
                            u'JPY': 0.509369, u'USD': 58.0833,
                            u'KZT': 0.173593, u'CHF': 58.1356,
                            u'NOK': 7.11186, u'BYR': 0.00294242,
                            u'TRY': 15.1654, u'UAH': 2.16124,
                            u'GBP': 75.9381, u'RUR': 1,
                            u'EUR': 67.5276},
                        'kz': {
                            u'JPY': 2.93, u'USD': 334.4, u'KZT': 1,
                            u'CHF': 334.6, u'NOK': 40.9686, u'BYR': 0.000169,
                            u'TRY': 88.55, u'UAH': 12.48, u'GBP': 437.9,
                            u'RUR': 5.75, u'EUR': 389.04
                        },
                        'ua': {
                            u'JPY': 0.235356, u'USD': 26.8758, u'KZT': 0.08037,
                            u'CHF': 26.8134, u'NOK': 3.27111, u'BYR': 0.00136,
                            u'TRY': 7.0308, u'UAH': 1, u'GBP': 35.1893,
                            u'RUR': 0.46271, u'EUR': 31.1893
                        }, 'com': {
                            u'JPY': 0.0075, u'USD': 0.8617, u'KZT': 0.00257043,
                            u'CHF': 0.8597, u'NOK': 0.1049,
                            u'BYR': 4.35736e-05, u'TRY': 0.2254,
                            u'UAH': 0.0320623, u'GBP': 1.1282,
                            u'RUR': 0.0147, u'EUR': 1
                        }, 'tr': {
                            u'JPY': 0.033354, u'USD': 3.7785,
                            u'KZT': 0.0112931, u'CHF': 3.8019,
                            u'NOK': 0.4659, u'BYR': 0.000194022,
                            u'TRY': 1, u'UAH': 0.1425, u'GBP': 4.9854,
                            u'RUR': 0.0653, u'EUR': 4.3988
                        }
                    }
                    break

        self._last_update = self._environment.now()

    def _check_fetch(self):
        cache_time = CurrencyRatesRepository.CACHE_TIME
        if (self._last_update - self._environment.now()) < cache_time:
            return
        self.fetch()

    def get_rates_for(self, national_version):
        self._check_fetch()

        if national_version not in self._index:
            self._logger.warn('Unknown national version %r', national_version)
            return {}

        return self._index[national_version]

    def get_base_currency(self, national_version):
        # type: (str) -> CurrencyModel
        base_currency_code = settings.AVIA_NATIONAL_CURRENCIES[national_version]
        return self._currency_repository.get_by_code(base_currency_code)


currency_rates_repository = CurrencyRatesRepository(
    currency_repository=currency_repository,
    rates_provider=converter,
    dohop_rates_provider=dohop_converter,
    environment=environment,
    logger=getLogger(__name__)
)
