from __future__ import absolute_import

from logging import Logger, getLogger  # noqa

from travel.avia.library.python.avia_data.models import CurrencyTranslation


class CurrencyTranslationModel(object):
    __slots__ = ('title', 'title_in', 'template', 'template_whole', 'template_cents')

    def __init__(self, title, title_in, template, template_whole, template_cents):
        # type: (unicode, unicode, unicode, unicode, unicode) -> None
        self.title = title
        self.title_in = title_in
        self.template = template
        self.template_whole = template_whole
        self.template_cents = template_cents

    def __repr__(self):
        return u'<CurrencyTranslationModel title={}>'.format(self.title)


EMPTY_TRANSLATION = CurrencyTranslationModel(
    title=u'',
    title_in=u'',
    template=u'',
    template_whole=u'',
    template_cents=u'',
)


class CurrencyTranslationRepository(object):
    ALL_LANGS = ['ru', 'uk', 'en', 'tr', 'de']
    FALLBACKS = {
        'uk': 'ru',
        'tr': 'en',
        'de': 'en'
    }

    def __init__(self, logger):
        # type: (Logger) -> None
        self._all_currencies = set()
        self._index = {}
        self._logger = logger

    def fetch(self):
        # type: () -> None

        translations = list(CurrencyTranslation.objects.values(
            'currency_id', 'lang__code', 'title', 'title_in',
            'template', 'template_whole', 'template_cents'
        ))

        index = {lang: {} for lang in self.ALL_LANGS}
        all_currencies = set()
        for t in translations:
            if t['lang__code'] not in index:
                self._logger.warn(
                    'Not support language %r in currency translation model [%r]',
                    t['lang__code'], t['currency_id']
                )
                continue
            all_currencies.add(t['currency_id'])
            index[t['lang__code']][t['currency_id']] = CurrencyTranslationModel(
                title=t['title'],
                title_in=t['title_in'],
                template=t['template'],
                template_whole=t['template_whole'],
                template_cents=t['template_cents']
            )

        for lang, translation_by_currency_id in index.iteritems():
            for currency_id in all_currencies:
                if currency_id not in translation_by_currency_id:
                    self._try_fill_translation_from_fallback(
                        lang=lang,
                        currency_id=currency_id,
                        index=index,
                        translation_by_currency_id=translation_by_currency_id
                    )

        self._index = index
        self._all_currencies = all_currencies

    def _try_fill_translation_from_fallback(self, lang, currency_id, index, translation_by_currency_id):
        self._logger.warn(
            'Translation for currency_id [%r] and'
            ' lang [%r] does not exists',
            currency_id,
            lang,
        )

        fallback_lang = CurrencyTranslationRepository.FALLBACKS.get(lang)
        if fallback_lang and currency_id in index[fallback_lang]:
            self._logger.info(
                'Use fallback translation [%r]'
                ' for currency_id [%r] and lang [%r]',
                fallback_lang,
                currency_id,
                lang,
            )

            translation_by_currency_id[currency_id] = (
                index[fallback_lang][currency_id]
            )
        else:
            self._logger.warn(
                'can not fill fallback for lang %r, currency_id %r',
                lang,
                currency_id,
            )

    def get_translation(self, currency_id, lang):
        # type: (int, str) -> CurrencyTranslationModel

        if currency_id not in self._all_currencies:
            self._logger.warn(
                'try get translation for unknown currency: %r', lang
            )
            return EMPTY_TRANSLATION

        if lang not in self._index:
            self._logger.warn('try get translation for unknown lang: %r', lang)
            return EMPTY_TRANSLATION

        lang_index = self._index[lang]

        if currency_id not in lang_index:
            self._logger.warn('can not find translation for currency %r', currency_id)
            return EMPTY_TRANSLATION

        return lang_index[currency_id]


currency_translation_repository = CurrencyTranslationRepository(
    logger=getLogger(__name__)
)
