from __future__ import absolute_import

import itertools
from typing import List  # noqa

from travel.avia.library.python.avia_data.models import AviaDirectionNational


class Direction(object):
    __slots__ = (
        'departure_settlement_id',
        'arrival_settlement_id',
        'direct_flights',
        'connecting_flights',
        'popularity',
        'national_version',
    )

    def __init__(self, departure_settlement_id, arrival_settlement_id,
                 direct_flights, connecting_flights,
                 popularity, national_version,
                 ):
        self.departure_settlement_id = int(departure_settlement_id)
        self.arrival_settlement_id = int(arrival_settlement_id)
        self.direct_flights = int(direct_flights)
        self.connecting_flights = int(connecting_flights)
        self.popularity = int(popularity)
        self.national_version = national_version


class DirectionRepository(object):
    LIMIT = 10

    def __init__(self):
        self._clear_cache()

    def _clear_cache(self):
        self._all = {}
        self._from = {}
        self._to = {}

    def pre_cache(self):
        # type: () -> None
        self._clear_cache()
        directions = [
            Direction(**d) for d in self._load_models()
        ]

        for direction in directions:
            if direction.national_version not in self._all:
                self._all[direction.national_version] = []
                self._from[direction.national_version] = {}
                self._to[direction.national_version] = {}

            cached = False

            if direction.departure_settlement_id not in self._from[direction.national_version]:
                self._from[direction.national_version][direction.departure_settlement_id] = []
            if len(self._from[direction.national_version][direction.departure_settlement_id]) < self.LIMIT:
                self._from[direction.national_version][direction.departure_settlement_id].append(direction)
                cached = True

            if direction.arrival_settlement_id not in self._to[direction.national_version]:
                self._to[direction.national_version][direction.arrival_settlement_id] = []
            if len(self._to[direction.national_version][direction.arrival_settlement_id]) < self.LIMIT:
                self._to[direction.national_version][direction.arrival_settlement_id].append(direction)
                cached = True

            if cached:
                self._all[direction.national_version].append(direction)

    @staticmethod
    def _load_models():
        # type: () -> list
        return AviaDirectionNational.objects.values(
            'departure_settlement_id', 'arrival_settlement_id',
            'direct_flights', 'connecting_flights',
            'popularity', 'national_version',
        ).order_by('-popularity')

    def get_all(self, national_version=None):
        # type: (str) -> List[Direction]
        if national_version:
            return self._all.get(national_version, [])
        return list(itertools.chain.from_iterable(self._all.itervalues()))

    def get_from_settlement(self, national_version, from_settlement_id):
        # type: (str, int) -> List[Direction]
        return self._from.get(national_version, {}).get(from_settlement_id, [])

    def get_to_settlement(self, national_version, to_settlement_id):
        # type: (str, int) -> List[Direction]
        return self._to.get(national_version, {}).get(to_settlement_id, [])


direction_repository = DirectionRepository()
