from __future__ import absolute_import

from typing import Optional, List  # noqa
from functools import partial
from collections import namedtuple
from operator import itemgetter
from travel.avia.library.python.common.utils.iterrecipes import group_by

from travel.avia.library.python.common.models.holidays import Holiday
from travel.avia.library.python.common.models.holidays import HolidayDirection

HolidayDirectionModel = namedtuple('HolidayDirection', ['from_settlement_pk', 'to_settlement_pk'])


class HolidayModel(object):
    __slots__ = (
        'pk', 'name_tanker_key', 'first_segment_first_day',
        'first_segment_last_day', 'second_segment_first_day',
        'second_segment_last_day', 'holiday_directions',
        'start_day', 'end_day',
    )

    def __init__(self,
                 pk,
                 name_tanker_key,
                 first_segment_first_day,
                 first_segment_last_day,
                 second_segment_first_day,
                 second_segment_last_day,
                 holiday_directions
                 ):
        self.pk = pk
        self.name_tanker_key = name_tanker_key
        self.first_segment_first_day = first_segment_first_day
        self.first_segment_last_day = first_segment_last_day
        self.second_segment_first_day = second_segment_first_day
        self.second_segment_last_day = second_segment_last_day
        self.holiday_directions = holiday_directions
        self.start_day = first_segment_first_day
        self.end_day = second_segment_last_day or first_segment_last_day

    @classmethod
    def from_db_record(cls, record, ghd):
        return HolidayModel(
            record['id'],
            record['name_tanker_key'],
            record['first_segment_first_day'],
            record['first_segment_last_day'],
            record['second_segment_first_day'],
            record['second_segment_last_day'],
            ghd.get(record['id'], [])
        )

    def __repr__(self):
        return u'<Holiday pk={}>'.format(self.pk)


class HolidayRepository(object):
    def __init__(self):
        # type: () -> None
        self._by_id = {}
        self._all = []

    def pre_cache(self):
        # type: () -> None

        active_holidays = Holiday.objects.filter(is_active=True)

        records = active_holidays.values(
            'id', 'name_tanker_key',
            'first_segment_first_day', 'first_segment_last_day',
            'second_segment_first_day', 'second_segment_last_day'
        )

        holiday_direction_records = HolidayDirection.objects.filter(
            holiday_id__in=active_holidays.values('id')
        ).values(
            'holiday_id', 'settlement_from_id', 'settlement_to_id'
        )

        grouped_holiday_directions = {
            holiday_id: [
                HolidayDirectionModel(
                    rec['settlement_from_id'],
                    rec['settlement_to_id']
                )
                for rec in group
            ]
            for holiday_id, group in group_by(
                holiday_direction_records, itemgetter('holiday_id')
            )
        }

        index = map(partial(HolidayModel.from_db_record, ghd=grouped_holiday_directions), records)

        self._by_id = {
            holiday.pk: holiday for holiday in index
        }

    def get_all(self):
        # type: () -> List[Holiday]
        return self._by_id.values()

    def get_by_id(self, model_id):
        # type: (int) -> Optional[Holiday]
        return self._by_id.get(model_id)


holiday_repository = HolidayRepository()
