from __future__ import absolute_import

from logging import Logger, getLogger  # noqa

from typing import Optional  # noqa

from travel.avia.library.python.avia_data.models import NationalVersion


class NationalVersionModel(object):
    __slots__ = ('pk', 'code')

    def __init__(self, pk, code):
        # type: (int, str) -> None
        self.pk = pk
        self.code = code

    def __repr__(self):
        return u'<NationalVersionModel pk={} code={}>'.format(
            self.pk, self.code
        )


class NationalVersionRepository(object):
    def __init__(self, logger):
        # type: (Logger) -> None
        self._id_to_model = {}
        self._code_to_model = {}
        self._all_models = []

        self._logger = logger

    def pre_cache(self):
        # type: () -> None

        id_to_model = {}
        code_to_model = {}
        all_models = []
        records = NationalVersion.objects.all().values_list(
            'id', 'code'
        )
        for model_id, code in records:
            m = NationalVersionModel(
                pk=model_id,
                code=code
            )
            id_to_model[model_id] = m
            code_to_model[code] = m
            all_models.append(m)

        self._id_to_model = id_to_model
        self._code_to_model = code_to_model

        self._all_models = all_models

    def id_to_code(self, model_id):
        # type: (int) -> Optional[str]
        if model_id not in self._id_to_model:
            self._logger.warn('Unknown national version id %r', model_id)
            return None

        return self._id_to_model[model_id].code

    def code_to_id(self, code):
        # type: (str) -> Optional[int]
        if code not in self._code_to_model:
            self._logger.warn('Unknown national version id %r', code)
            return None

        return self._code_to_model[code].pk

    def get_all(self):
        return self._all_models


national_version_repository = NationalVersionRepository(
    logger=getLogger(__name__)
)
