from __future__ import absolute_import

from collections import defaultdict
from typing import List  # noqa

from travel.avia.library.python.avia_data.models import PopularPartners, PopularPartnersByRoute


class PartnerPopularityModel(object):
    __slots__ = ('partner_id', 'score')

    def __init__(self, partner_id, score):
        # type: (int, int) -> None
        self.partner_id = partner_id
        self.score = score


class PartnerPopularityRepository(object):
    def __init__(self, ):
        # type: () -> None
        self._index = {}

    def pre_cache(self):
        # type: () -> None

        index = defaultdict(list)
        records = PopularPartners.objects.all().values(
            'national_version_id', 'partner_id', 'score'
        )

        for record in records:
            index[record['national_version_id']].append(
                PartnerPopularityModel(record['partner_id'], record['score'])
            )

        index = dict(index)
        for national_version_id in index:
            index[national_version_id] = sorted(
                index[national_version_id],
                key=lambda x: (x.score, x.partner_id),
                reverse=True
            )

        self._index = index

    def get_for(self, national_version_id):
        # type: (int) -> List[PartnerPopularityModel]
        return list(self._index.get(national_version_id, []))


class RoutePartnerPopularityRepository(object):
    def __init__(self):
        # type: () -> None
        self._index = {}

    def pre_cache(self):
        # type: () -> None

        index = defaultdict(list)
        records = PopularPartnersByRoute.objects.all().values(
            'national_version_id', 'partner_id', 'score',
            'from_type', 'from_id', 'to_type', 'to_id'
        )

        for r in records:
            index[(
                r['national_version_id'],
                r['from_type'], r['from_id'],
                r['to_type'], r['to_id']
            )].append(
                PartnerPopularityModel(r['partner_id'], r['score'])
            )

        index = dict(index)
        for key in index:
            index[key] = sorted(
                index[key],
                key=lambda x: (x.score, x.partner_id),
                reverse=True
            )

        self._index = index

    def get_for(self, national_version_id, from_type, from_id, to_type, to_id):
        # type: (int, int, int, int, int) -> List[PartnerPopularityModel]
        return list(self._index.get(
            (national_version_id, from_type, from_id, to_type, to_id), []
        ))


partner_popularity_repository = PartnerPopularityRepository()
route_partner_popularity_repository = RoutePartnerPopularityRepository()
