# coding=utf-8
from __future__ import absolute_import

from collections import defaultdict

from django.conf import settings
from typing import List, Optional  # noqa

from travel.avia.library.python.avia_data.models import AviaRecipe
from travel.avia.backend.repository.helpers import NationalBox


class RecipeModel(object):
    __slots__ = (
        'pk', 'week_days', 'date_start', 'date_end',
        'backward_date_start', 'backward_date_end',
        'show_start', 'show_end', 'show_year',
        'order', 'slug', 'enabled_by_nv', 'from_region_ids', 'from_settlement_ids',
        'region_ids', 'settlement_ids', 'country_ids',
        'meta_title_key', '_h1', '_meta_title', '_title', '_meta_description',
    )

    def __init__(self, pk, week_days, date_start, date_end,
                 backward_date_start, backward_date_end,
                 show_start, show_end, show_year,
                 order, slug,
                 enabled_by_nv,
                 from_region_ids, from_settlement_ids,
                 region_ids, settlement_ids, country_ids,
                 meta_title_key,
                 title_by_lang,
                 meta_description_by_lang,
                 meta_title_by_lang,
                 h1_by_lang
                 ):
        self.pk = pk
        self.week_days = week_days
        self.date_start = date_start
        self.date_end = date_end
        self.backward_date_start = backward_date_start
        self.backward_date_end = backward_date_end
        self.show_start = show_start
        self.show_end = show_end
        self.show_year = show_year
        self.order = order
        self.slug = slug
        self.meta_title_key = meta_title_key
        self.enabled_by_nv = enabled_by_nv

        self.from_region_ids = from_region_ids
        self.from_settlement_ids = from_settlement_ids

        self.settlement_ids = settlement_ids
        self.region_ids = region_ids
        self.country_ids = country_ids

        self._h1 = h1_by_lang
        self._meta_title = meta_title_by_lang
        self._title = title_by_lang
        self._meta_description = meta_description_by_lang

    def get_h1(self, lang):
        return self._h1.get(lang) or u''

    def get_title(self, lang):
        return self._title.get(lang) or u''

    def get_meta_description(self, lang):
        return self._meta_description.get(lang) or u''

    def get_meta_title(self, lang):
        return self._meta_title.get(lang) or u''


popular_recipe = RecipeModel(
    pk=0,
    week_days=None,
    date_start=None,
    date_end=None,
    backward_date_start=None,
    backward_date_end=None,
    show_start=None,
    show_end=None,
    show_year=None,
    order=0,
    slug=None,
    meta_title_key=None,

    from_settlement_ids=set(),
    from_region_ids=set(),
    settlement_ids=set(),
    region_ids=set(),
    country_ids=set(),

    enabled_by_nv=NationalBox(
        dict.fromkeys(settings.AVIA_NATIONAL_VERSIONS, True)
    ),

    title_by_lang=dict.fromkeys(
        settings.AVIA_NATIONAL_VERSIONS,
        u'Популярные направления'
    ),
    meta_description_by_lang={},
    meta_title_by_lang={},
    h1_by_lang={}
)


class RecipesRepository(object):
    LANGS = ['ru', 'en', 'tr', 'uk']
    NATIONAL_VERSIONS = ['ru', 'ua', 'tr', 'com', 'kz']

    def __init__(self):
        self._recipes = []
        self._recipe_by_id = {}

    def pre_cache(self):
        fields = [
            'id', 'week_days', 'date_start', 'date_end', 'backward_date_start',
            'backward_date_end', 'show_start', 'show_end', 'show_year',
            'order', 'slug', 'meta_title_key',
        ]
        fields += [
            'enabled_{}'.format(nv)
            for nv in RecipesRepository.NATIONAL_VERSIONS
        ]
        fields += [
            'title_{}'.format(lang)
            for lang in RecipesRepository.LANGS
        ]
        fields += [
            'meta_description_{}'.format(lang)
            for lang in RecipesRepository.LANGS
        ]
        fields += [
            'meta_title_{}'.format(lang)
            for lang in RecipesRepository.LANGS
        ]
        fields += [
            'h1_{}'.format(lang)
            for lang in RecipesRepository.LANGS
        ]

        recipes = [popular_recipe, ]
        recipe_by_id = {
            popular_recipe.pk: popular_recipe,
        }

        many_to_many_fields = [
            'from_settlements', 'from_regions',
            'countries', 'regions', 'settlements'
        ]
        recipe_id_to_m2m = defaultdict(lambda: defaultdict(set))

        for record in AviaRecipe.objects.values(*(['id'] + many_to_many_fields)):
            for field in [f for f in many_to_many_fields if record[f]]:
                recipe_id_to_m2m[record['id']][field].add(record[field])

        for raw_recipe in AviaRecipe.objects.values(*fields):
            pk = raw_recipe['id']

            many_to_many = recipe_id_to_m2m[pk]

            m = RecipeModel(
                pk=pk,
                week_days=raw_recipe['week_days'],
                date_start=raw_recipe['date_start'],
                date_end=raw_recipe['date_end'],
                backward_date_start=raw_recipe['backward_date_start'],
                backward_date_end=raw_recipe['backward_date_end'],
                show_start=raw_recipe['show_start'],
                show_end=raw_recipe['show_end'],
                show_year=raw_recipe['show_year'],
                order=raw_recipe['order'],
                slug=raw_recipe['slug'],
                meta_title_key=raw_recipe['meta_title_key'],

                from_settlement_ids=many_to_many['from_settlements'],
                from_region_ids=many_to_many['from_regions'],
                settlement_ids=many_to_many['settlements'],
                region_ids=many_to_many['regions'],
                country_ids=many_to_many['countries'],

                enabled_by_nv=NationalBox(
                    {
                        lang: raw_recipe['enabled_{}'.format(lang)]
                        for lang in RecipesRepository.NATIONAL_VERSIONS
                    }
                ),

                title_by_lang={
                    lang: raw_recipe['title_{}'.format(lang)]
                    for lang in RecipesRepository.LANGS
                },
                meta_description_by_lang={
                    lang: raw_recipe['meta_description_{}'.format(lang)]
                    for lang in RecipesRepository.LANGS
                },
                meta_title_by_lang={
                    lang: raw_recipe['meta_title_{}'.format(lang)]
                    for lang in RecipesRepository.LANGS
                },
                h1_by_lang={
                    lang: raw_recipe['h1_{}'.format(lang)]
                    for lang in RecipesRepository.LANGS
                }
            )

            recipes.append(m)
            recipe_by_id[m.pk] = m

        self._recipe_by_id = recipe_by_id
        self._recipes = recipes

    def get_all(self):
        # type: () -> List[RecipeModel]
        return self._recipes

    def get_by_id(self, recipe_id):
        # type: (int) -> Optional[RecipeModel]
        return self._recipe_by_id.get(recipe_id)


recipes_repository = RecipesRepository()
