from __future__ import absolute_import

from typing import Optional, List  # noqa

from unidecode import unidecode

from travel.avia.library.python.common.models.geo import Region
from travel.avia.backend.repository.translations import TranslatedTitleRepository, translated_title_repository  # noqa


class RegionModel(object):
    __slots__ = (
        '_translated_title_repository',
        'pk', 'point_key', '_title_id', 'geo_id', 'country_id',
    )

    def __init__(self, translated_title_repository, pk, title_id, geo_id, country_id):
        # type: (TranslatedTitleRepository, int, int, Optional[int], Optional[int]) -> None
        self._translated_title_repository = translated_title_repository

        self.pk = pk
        self.point_key = u'r{}'.format(pk)

        self._title_id = title_id

        self.geo_id = geo_id
        self.country_id = country_id

    def get_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get(self._title_id, lang)

    def get_genitive_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get_genitive(
            self._title_id, lang
        )

    def get_accusative_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get_accusative(
            self._title_id, lang
        )

    def get_url_title(self, lang):
        # type: (str) -> str
        return unidecode(self.get_title(lang=lang)).replace("'", '')

    def prepare(self, lang):
        # type: (str) -> dict
        return {
            'id': self.pk,
            'title': self.get_title(lang),
            'geoId': self.geo_id,
        }

    def __repr__(self):
        return u"<RegionModel pk={} title_ru={} title_en={}>".format(
            self.pk,
            self.get_title('ru'),
            self.get_title('en'),
        )


class RegionRepository(object):
    def __init__(self, translated_title_repository):
        # type: (TranslatedTitleRepository) -> None
        self._translated_title_repository = translated_title_repository

        self._index = {}
        self._geo_id_index = {}

    def _load_db_models(self):
        # type: () -> List[dict]
        fields = [
            'id',
            'new_L_title_id',
            '_geo_id',
            'country_id',
        ]

        return list(Region.objects.values(*fields))

    def pre_cache(self):
        # type: () -> None
        regions = self._load_db_models()

        title_ids = set(r['new_L_title_id'] for r in regions)
        self._translated_title_repository.fetch(title_ids)

        index = {}
        geo_id_index = {}
        for s in regions:
            pk = s['id']
            geo_id = s['_geo_id']

            m = RegionModel(
                translated_title_repository=self._translated_title_repository,

                pk=pk,
                title_id=s['new_L_title_id'],

                geo_id=geo_id,
                country_id=s['country_id'],
            )

            index[pk] = m
            if geo_id is not None:
                geo_id_index[geo_id] = m

        self._index = index
        self._geo_id_index = geo_id_index

    def get(self, region_id):
        # type: (int) -> Optional[RegionModel]
        return self._index.get(region_id)

    def get_by_geo_id(self, geo_id):
        # type: (int) -> Optional[RegionModel]
        return self._geo_id_index.get(geo_id)

    def get_all(self):
        # type: () -> List[RegionModel]
        return self._index.values()


region_repository = RegionRepository(translated_title_repository)
