from __future__ import absolute_import

import datetime  # noqa
from operator import itemgetter

from typing import Optional, List, Generator, Tuple  # noqa
from unidecode import unidecode

from geoindex import GeoGridIndex, GeoPoint

from travel.avia.library.python.avia_data.models import AviaDirectionNational
from travel.avia.library.python.common.models.geo import Settlement, CityMajority
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.date import get_pytz
from travel.avia.backend.repository.translations import TranslatedTitleRepository, translated_title_repository  # noqa


class SettlementModel(object):
    __slots__ = (
        '_translated_title_repository',
        'pk', 'id', 'point_key', '_title_id',
        'iata', 'sirena', 'geo_id',
        'country_id', 'region_id', 'is_disputed_territory',
        'majority_id', 'pytz', 'utcoffset', 'latitude', 'longitude',
    )

    def __init__(self, translated_title_repository,
                 pk, title_id,
                 iata, sirena, geo_id,
                 country_id, region_id, is_disputed_territory,
                 majority_id, pytz, utcoffset, latitude, longitude):
        # type: (TranslatedTitleRepository, int, int, Optional[str], Optional[str], Optional[int], Optional[int], Optional[int], bool, int, datetime.tzinfo, int, int, int) -> None
        self._translated_title_repository = translated_title_repository

        self.pk = pk
        self.id = pk  # for backward compatibility
        self.point_key = u'c{}'.format(pk)

        self._title_id = title_id

        self.iata = iata
        self.sirena = sirena
        self.geo_id = geo_id

        self.country_id = country_id
        self.region_id = region_id

        self.is_disputed_territory = is_disputed_territory
        self.majority_id = majority_id

        self.pytz = pytz
        self.utcoffset = utcoffset
        self.latitude = latitude
        self.longitude = longitude

    def get_title(self, lang):
        # type: (str) -> str
        if self._translated_title_repository:
            return (
                self._translated_title_repository.get(self._title_id, lang) or u''
            )
        return "get_title {} for {}".format(self._title_id, lang)

    def get_genitive_title(self, lang):
        # type: (str) -> unicode
        if self._translated_title_repository:
            return (
                self._translated_title_repository.get_genitive(self._title_id, lang) or u''
            )

        return "get_genitive_title {} for {}".format(self._title_id, lang)

    def get_accusative_title(self, lang):
        # type: (str) -> unicode
        if self._translated_title_repository:
            return (
                self._translated_title_repository.get_accusative(self._title_id, lang) or u''
            )
        return "get_accusative_title {} for {}".format(self._title_id, lang)

    def get_locative_title(self, lang):
        # type: (str) -> unicode
        if self._translated_title_repository:
            return (
                self._translated_title_repository.get_locative(self._title_id, lang) or u''
            )

        return "get_locative_title {} for {}".format(self._title_id, lang)

    def get_url_title(self, lang):
        # type: (str) -> str
        return unidecode(self.get_title(lang=lang)).replace("'", '')

    def get_phrase_from(self, lang):
        # type: (str) -> unicode
        if lang not in ('ru', 'uk'):
            return self.get_title(lang)

        title = self.get_genitive_title(lang)

        if not title:
            return self.get_title(lang)

        return title

    def get_phrase_to(self, lang):
        # type: (str) -> unicode
        if lang not in ('ru', 'uk'):
            return self.get_title(lang)

        title = self.get_accusative_title(lang)

        if not title:
            return self.get_title(lang)

        return title

    def get_phrase_in(self, lang):
        # type: (str) -> unicode
        if lang not in ('ru', 'uk'):
            return self.get_title(lang)

        title = self.get_locative_title(lang)

        if not title:
            return self.get_title(lang)

        return title

    def prepare(self, lang):
        # type: (str) -> dict
        return {
            'id': self.pk,
            'geoId': self.geo_id,
            'countryId': self.country_id,
            'iata': self.iata,
            'sirena': self.sirena,
            'title': self.get_title(lang),
            'phraseFrom': self.get_phrase_from(lang),
            'phraseIn': self.get_phrase_in(lang),
            'phraseTo': self.get_phrase_to(lang),
            'urlTitle': unidecode(self.get_title(lang)).replace("'", ''),
            'longitude': self.longitude,
            'latitude': self.latitude,
        }

    def __repr__(self):
        d = u"<SettlementModel pk={} title_ru={} title_en={}>".format(
            unicode(self.pk),
            self.get_title('ru'),
            self.get_title('en')
        )

        return d


class SettlementRepository(object):
    def __init__(self, translated_title_repository, environment):
        # type: (TranslatedTitleRepository, any) -> None
        self._translated_title_repository = translated_title_repository
        self._environment = environment

        self._index = {}

        self._by_geo_id = {}

        self._country_capital_cities = {}
        self._region_capital_cities = {}

    def _load_db_models(self):
        # type: () -> List[dict]
        fields = [
            'id',
            'new_L_title_id',
            'iata', 'sirena_id', '_geo_id',
            'country_id', 'region_id', '_disputed_territory',
            'majority_id', 'time_zone', 'latitude', 'longitude'
        ]

        return list(Settlement.objects.values(*fields))

    def pre_cache(self):
        # type: () -> None
        settlements = self._load_db_models()

        title_ids = set(c['new_L_title_id'] for c in settlements)
        self._translated_title_repository.fetch(title_ids)

        index = {}
        by_geo_id = {}
        country_capital_cities = {}
        region_capital_cities = {}

        now = self._environment.now_aware()
        for s in settlements:
            pk = s['id']
            geo_id = s['_geo_id']
            pytz = get_pytz(s['time_zone'])
            utcoffset = int(now.astimezone(pytz).utcoffset().total_seconds())

            m = SettlementModel(
                translated_title_repository=self._translated_title_repository,

                pk=pk,

                title_id=s['new_L_title_id'],

                iata=s['iata'],
                sirena=s['sirena_id'],
                geo_id=geo_id,

                country_id=s['country_id'],
                region_id=s['region_id'],

                is_disputed_territory=s['_disputed_territory'],
                majority_id=s['majority_id'],
                pytz=pytz,
                utcoffset=utcoffset,
                latitude=s['latitude'],
                longitude=s['longitude']
            )

            index[pk] = m
            if geo_id is not None:
                by_geo_id[geo_id] = m

            if m.country_id and m.majority_id == CityMajority.CAPITAL_ID:
                country_capital_cities[m.country_id] = m

            if m.region_id:
                if m.majority_id == CityMajority.REGION_CAPITAL_ID or m.majority_id == CityMajority.CAPITAL_ID:
                    region_capital_cities[m.region_id] = m

        self._index = index
        self._by_geo_id = by_geo_id
        self._country_capital_cities = country_capital_cities
        self._region_capital_cities = region_capital_cities

    def get(self, settlement_id):
        # type: (int) -> Optional[SettlementModel]
        return self._index.get(settlement_id)

    def get_by_geo_id(self, geo_id):
        # type: (int) -> Optional[SettlementModel]
        return self._by_geo_id.get(geo_id)

    def get_country_capital(self, country_id):
        # type: (int) -> Optional[SettlementModel]
        return self._country_capital_cities.get(country_id)

    def get_region_capital(self, region_id):
        # type: (int) -> Optional[SettlementModel]
        return self._region_capital_cities.get(region_id)

    def get_all(self):
        # type: () -> List[SettlementModel]
        return self._index.values()


class SettlementGeoIndex(object):
    def __init__(self, repository):
        # type: (SettlementRepository) -> None
        self._repository = repository
        self._geo_index = GeoGridIndex(precision=2)

    def pre_cache(self):
        self._geo_index = GeoGridIndex(precision=2)
        settlement_ids = set(
            Settlement.objects
            .filter(hidden=False, type_choices__contains='plane')
            .values_list('id', flat=True)
        )
        arrival_settlements = set(
            AviaDirectionNational.objects
            .values_list('arrival_settlement', flat=True)
            .distinct()
        )

        for settlement_id in settlement_ids:
            settlement = self._repository.get(settlement_id)
            if (
                settlement.latitude and settlement.longitude
                and settlement.pk in arrival_settlements
            ):
                self._geo_index.add_point(self._geo_point(settlement))

    def get_nearest_with_distance(self, settlement_id, distance, include_itself=False):
        # type: (int, int, bool) -> Generator[Tuple[SettlementModel, int]]
        settlement = self._repository.get(settlement_id)
        if not settlement:
            return
        center = self._geo_point(settlement)
        nearest = self._geo_index.get_nearest_points(center, distance, 'km')
        for point, distance in sorted(nearest, key=itemgetter(1)):
            if point.ref == settlement_id and not include_itself:
                continue
            yield self._repository.get(point.ref), distance

    def get_nearest(self, settlement_id, distance, include_itself=False):
        # type: (int, int, bool) -> List[SettlementModel]
        return map(
            itemgetter(0),
            self.get_nearest_with_distance(settlement_id, distance, include_itself)
        )

    def get_nearest_by_coordinates(self, latitude, longitude, distance):
        # type: (float, float, int) -> List[SettlementModel]
        original_point = GeoPoint(latitude, longitude)
        nearest = self._geo_index.get_nearest_points(original_point, distance, 'km')
        return [
            self._repository.get(point.ref)
            for point, _distance in sorted(nearest, key=itemgetter(1))
        ]

    @staticmethod
    def _geo_point(settlement):
        return GeoPoint(
            settlement.latitude, settlement.longitude, ref=settlement.pk
        )


settlement_repository = SettlementRepository(
    translated_title_repository,
    environment
)
settlement_geo_index = SettlementGeoIndex(settlement_repository)
