from __future__ import absolute_import

import random

from typing import Optional, List  # noqa
from logging import Logger, getLogger  # noqa

from travel.avia.library.python.avia_data.models import SettlementBigImage


class SettlementBigImageModel(object):
    __slots__ = (
        'pk', 'settlement_id', 'url2',
    )

    def __init__(self, pk, settlement_id, url2):
        # type: (int, Optional[int], str) -> None
        self.pk = pk
        self.settlement_id = settlement_id
        self.url2 = url2

    def is_default_image(self):
        # type: () -> bool
        return self.settlement_id is not None

    def __repr__(self):
        return u"<SettlementBigImageModel pk={} settlement_id={} url2={}>".format(
            self.pk,
            self.settlement_id,
            self.url2
        )


class SettlementBigImageRepository(object):
    def __init__(self, logger):
        # type: (Logger) -> None

        self._index = {}
        self._defaults = []
        self._logger = logger

    def _load_db_models(self):
        # type: () -> List[dict]
        fields = [
            'id',
            'settlement_id',
            'url2',
        ]

        return list(SettlementBigImage.objects.values(*fields))

    def pre_cache(self):
        # type: () -> None
        images = self._load_db_models()

        index = {}
        defaults = []
        for img in images:
            pk = img['id']
            settlement_id = img.get('settlement_id')
            url2 = img['url2']

            entry = SettlementBigImageModel(
                pk=pk,
                settlement_id=settlement_id,
                url2=url2,
            )

            if settlement_id:
                index[settlement_id] = entry
            else:
                defaults.append(entry)

        self._index = index
        self._defaults = defaults

    def get(self, settlement_id):
        # type: (int) -> Optional[SettlementBigImageModel]
        return self._index.get(settlement_id)

    def get_image_or_default_url(self, settlement_id):
        # type: (int) -> Optional[str]
        image = self._index.get(settlement_id) or self.get_random_default()
        return image.url2 if image else None

    def get_random_default(self):
        # type: () -> Optional[SettlementBigImageModel]

        if not self._defaults:
            self._logger.warn('Unable to fetch the default image for the travel recipe.')
            return None

        return random.choice(self._defaults)

    def get_defaults(self):
        # type: () -> List[SettlementBigImageModel]
        return self._defaults

    def get_all(self):
        # type: () -> List[SettlementBigImageModel]
        return self._index.values()


settlement_big_image_repository = SettlementBigImageRepository(logger=getLogger(__name__))
