from __future__ import absolute_import

from collections import defaultdict
from typing import Optional, List  # noqa
from unidecode import unidecode

from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.date import get_pytz
from travel.avia.library.python.common.models.geo import Station, StationCode
from travel.avia.backend.repository.translations import TranslatedTitleRepository, translated_title_repository  # noqa


class StationModel(object):
    __slots__ = (
        '_translated_title_repository',
        'pk', 'point_key', '_title_id', '_popular_title_id', 'iata', 'sirena',
        'settlement_id', 'country_id', 'region_id',
        'longitude', 'latitude', 'time_zone', 'time_zone_utc_offset',
        'station_type_id', 'transport_type',
    )

    def __init__(self, translated_title_repository,
                 pk, title_id, popular_title_id, iata, sirena,
                 settlement_id, country_id, region_id,
                 longitude, latitude, time_zone, time_zone_utc_offset,
                 station_type_id, transport_type,
                 ):
        # type: (TranslatedTitleRepository, int, int, int, str, str, int, int, int, float, float, str, int, int) -> None
        self._translated_title_repository = translated_title_repository

        self.pk = pk
        self.point_key = u's{}'.format(pk)

        self._title_id = title_id
        self._popular_title_id = popular_title_id

        self.iata = iata
        self.sirena = sirena

        self.settlement_id = settlement_id
        self.country_id = country_id
        self.region_id = region_id

        self.longitude = longitude
        self.latitude = latitude

        self.time_zone = time_zone
        self.time_zone_utc_offset = time_zone_utc_offset

        self.station_type_id = station_type_id
        self.transport_type = transport_type

    def get_title(self, lang):
        # type: (str) -> str
        return self._translated_title_repository.get(self._title_id, lang)

    def get_url_title(self, lang):
        # type: (str) -> str
        title = self.get_title(lang=lang)
        return unidecode(title.replace("'", '')) if title else ''

    def get_accusative_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get_accusative(
            self._title_id, lang
        )

    def get_genitive_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get_genitive(
            self._title_id, lang
        )

    def get_locative_title(self, lang):
        # type: (str) -> unicode
        return self._translated_title_repository.get_locative(
            self._title_id, lang
        )

    def get_phrase_to(self, lang):
        # type: (str) -> unicode
        if lang not in ('ru', 'uk'):
            return self.get_title(lang)

        title = self.get_accusative_title(lang)

        if not title:
            return self.get_title(lang)

        return title

    def get_phrase_from(self, lang):
        # type: (str) -> unicode
        if lang not in ('ru', 'uk'):
            return self.get_title(lang)

        title = self.get_genitive_title(lang)

        if not title:
            return self.get_title(lang)

        return title

    def get_phrase_in(self, lang):
        # type: (str) -> unicode
        if lang not in ('ru', 'uk'):
            return self.get_title(lang)

        title = self.get_locative_title(lang)

        if not title:
            return self.get_title(lang)

        return title

    def get_popular_title(self, lang):
        # type: (str) -> str
        title = self._translated_title_repository.get(self._popular_title_id, lang)
        if not title:
            title = self.get_title(lang)

        return title

    def __repr__(self):
        return u'<StationModel pk={} title_ru={} title_en={}>'.format(
            self.pk,
            self.get_title('ru'),
            self.get_title('en'),
        )


class StationRepository(object):
    def __init__(self, translated_title_repository):
        # type: (TranslatedTitleRepository) -> None
        self._translated_title_repository = translated_title_repository

        self._index = {}
        self._aeroexpress_by_settlement_id = {}

    @staticmethod
    def _load_db_models():
        # type: () -> List[dict]
        fields = [
            'id',
            'new_L_title_id', 'new_L_popular_title_id',
            'sirena_id', 'station_type_id',
            'country_id', 'settlement_id', 'region_id',
            'type_choices', 'longitude', 'latitude',
            'time_zone', 't_type_id',
        ]

        return Station.objects.values(*fields)

    @staticmethod
    def _load_station_iata_codes():
        return {
            raw['station_id']: raw['code']
            for raw in StationCode.objects.filter(system__code='iata').values(
                'code', 'station_id'
            )
        }

    def pre_cache(self):
        # type: () -> None
        stations = self._load_db_models()

        self._translated_title_repository.fetch(
            {c['new_L_title_id'] for c in stations} | {c['new_L_popular_title_id'] for c in stations}
        )

        station_id_to_iata = self._load_station_iata_codes()

        index = {}
        aeroexpress_by_settlement_id = defaultdict(list)
        for s in stations:
            pk = s['id']
            m = StationModel(
                translated_title_repository=self._translated_title_repository,

                pk=pk,

                title_id=s['new_L_title_id'],
                popular_title_id=s['new_L_popular_title_id'],

                iata=station_id_to_iata.get(pk),
                sirena=s['sirena_id'],

                country_id=s['country_id'],
                settlement_id=s['settlement_id'],
                region_id=s['region_id'],
                longitude=s['longitude'],
                latitude=s['latitude'],
                time_zone=s['time_zone'],
                time_zone_utc_offset=int(
                    environment.now_aware().astimezone(get_pytz(s['time_zone'])).utcoffset().total_seconds()
                ) / 60,
                station_type_id=s['station_type_id'],
                transport_type=s['t_type_id'],
            )
            index[pk] = m

            type_choices = s['type_choices'] or ''
            if 'aeroexp' in type_choices and m.settlement_id:
                aeroexpress_by_settlement_id[m.settlement_id] = m

        self._index = index
        self._aeroexpress_by_settlement_id = aeroexpress_by_settlement_id

    def get(self, station_id, transport_type=None):
        # type: (int, int) -> Optional[StationModel]
        res = self._index.get(station_id)
        if transport_type and res and res.transport_type != transport_type:
            return None
        return res

    def get_all(self, transport_type=None):
        # type: (int) -> List[StationModel]
        if transport_type:
            return filter(lambda x: x.transport_type == transport_type, self._index.values())

        return self._index.values()

    def get_aeroexpress_for_settlement(self, settlement_pk):
        # type: (int) -> List[StationModel]
        return self._aeroexpress_by_settlement_id.get(settlement_pk, [])


station_repository = StationRepository(translated_title_repository)
