# -*- coding: utf-8 -*-
from __future__ import absolute_import

from typing import Optional  # noqa

from travel.avia.library.python.common.models.geo import StationType
from travel.avia.backend.repository.translations import TranslatedTitleRepository, translated_title_repository  # noqa


class StationTypeModel(object):
    __slots__ = ('_translated_title_repository', 'pk', '_title_id', '_prefix_id')

    def __init__(self, translated_title_repository, pk, title_id, prefix_id):
        # type: (TranslatedTitleRepository, int, int, int) -> None
        self._translated_title_repository = translated_title_repository
        self.pk = pk
        self._title_id = title_id
        self._prefix_id = prefix_id

    def get_title(self, lang):
        # type: (str) -> str
        return self._translated_title_repository.get(self._title_id, lang)

    def get_prefix(self, lang):
        # type: (str) -> str
        return self._translated_title_repository.get(self._prefix_id, lang)

    def __repr__(self):
        return u"<StationTypeModel pk={} title_ru={} title_en={}>".format(
            self.pk,
            self.get_title('ru'),
            self.get_title('en'),
        )


class StationTypeRepository(object):
    def __init__(self, translated_title_repository):
        # type: (TranslatedTitleRepository) -> None
        self._translated_title_repository = translated_title_repository
        self._by_id = {}

    def pre_cache(self):
        # type: () -> None
        types = StationType.objects.values('id', 'new_L_name_id', 'new_L_prefix_id')

        self._translated_title_repository.fetch(
            {t['new_L_name_id'] for t in types} | {t['new_L_prefix_id'] for t in types}
        )

        for t in types:
            self._by_id[t['id']] = StationTypeModel(
                translated_title_repository=self._translated_title_repository,
                pk=t['id'],
                title_id=t['new_L_name_id'],
                prefix_id=t['new_L_prefix_id'],
            )

    def get(self, type_id):
        # type: (int) -> Optional[StationTypeModel]
        return self._by_id.get(type_id)


station_type_repository = StationTypeRepository(translated_title_repository)
