from __future__ import absolute_import

from typing import Set, List  # noqa

from travel.avia.library.python.common.models.translations import TranslatedTitle


class TranslatedTitleRepository(object):
    ALL_LANGS = ['ru', 'uk', 'en', 'tr', 'de']
    FALLBACKS = {
        'uk': 'ru',
        'tr': 'en',
        'de': 'en'
    }
    CASES = ['ru_genitive', 'ru_accusative', 'uk_accusative', 'ru_locative']

    def __init__(self):
        # type: () -> None
        self.reset()

    def reset(self):
        # type: () -> None
        self._fetched_ids = set()
        self._translations = {}

    def fetch(self, ids):
        # type: (Set[int]) -> None
        translations = self._load_models(ids - self._fetched_ids)

        for t in translations:
            self._translations[t['id']] = {}
            for case in self.CASES:
                self._translations[t['id']][case] = t[case]

            for lang in self.ALL_LANGS:
                title = t.get(self.nominative_key(lang))
                if not title and lang in self.FALLBACKS:
                    title = t.get(self.nominative_key(self.FALLBACKS[lang]))
                self._translations[t['id']][self.nominative_key(lang)] = title

        self._fetched_ids |= ids

    def _load_models(self, new_ids):
        # type: (Set[int]) -> List[dict]
        fields = ['id'] + self.CASES + [self.nominative_key(lang) for lang in self.ALL_LANGS]

        return list(
            TranslatedTitle.objects.filter(id__in=new_ids).values(*fields)
        )

    def get(self, translation_id, lang):
        # type: (int, str) -> unicode
        return self._translations.get(translation_id, {}).get(self.nominative_key(lang), u'')

    def get_genitive(self, translation_id, lang):
        # type: (int, str) -> unicode
        if lang != 'ru':
            return u''

        return self._translations.get(translation_id, {}).get(lang + '_genitive', u'')

    def get_accusative(self, translation_id, lang):
        # type: (int, str) -> unicode
        if lang not in {'ru', 'uk'}:
            return u''

        return self._translations.get(translation_id, {}).get(lang + '_accusative', u'')

    def get_locative(self, translation_id, lang):
        # type: (int, str) -> unicode
        if lang != 'ru':
            return u''

        return self._translations.get(translation_id, {}).get(lang + '_locative', u'')

    @classmethod
    def nominative_key(cls, lang):
        return '{}_nominative'.format(lang)


translated_title_repository = TranslatedTitleRepository()
